<?php

namespace App\Classes\ClientSPS;

use Illuminate\Support\Facades\Auth;

use App\Client;
use App\Multisafe;


// classe incluída em portugues para não confundir com a classe do model
class ClientSpsService
{
    private $hierarquia; // M = Mãe ; F = Filho
    public $clientId;
    private $taxnumber;
    private $clientModel;
    private $multisafeModel;
    private $saldoCompensar; // saldo que o cliente tem no cofre
    private $saldoReal; // saldo já compensado, depositado no banco
    private $nickname;

    private $balanceusagepercentage;
    private $balanceusage;

    function __construct($_taxnumber)
    {
        $this->taxnumber = $this->onlyNumbersTaxnumber($_taxnumber);
        $this->clientId = $this->resolveClientId();
        $this->clientModel = $this->getModelClient();
        $this->hierarquia = $this->identifyHierarchy();

        $this->balanceusagepercentage = $this->clientModel->balanceusagepercentage;
        $this->balanceusage = $this->clientModel->balanceusage;
    }

    function executarTestesDEV($_valorTotalOperacao)
    {
        $result = array(
            'nome'              => $this->getNickname(),
            'clientId'          => $this->clientId,
            'taxnumber'         => $this->taxnumber,
            'client_id'         => $this->clientId,
            'hierarquia'        => $this->hierarquia,
            'saldoReal'         => $this->getSaldoReal(),
            'saldoCompensar'    => $this->getSaldoCompensar(),
            'limiteTransferInterna' => $this->calculaLimiteDaOperacao('TRANSFERENCIA_INTERNA', 1),
            'limiteTransfer'    => $this->calculaLimiteDaOperacao('TRANSFERENCIA_EXTERNA', 1),
            'limiteBoleto'      => $this->calculaLimiteDaOperacao('PAGAMENTO_BOLETO', 1),
            'valorTotalOperacao' => $_valorTotalOperacao,
            'adiantarTransfInterna' => $this->calculaSaldoAAdiantarParaOperacao('TRANSFERENCIA_INTERNA', $_valorTotalOperacao),
            'adiantarTransfer'  => $this->calculaSaldoAAdiantarParaOperacao('TRANSFERENCIA_EXTERNA', $_valorTotalOperacao),
            'adiantarBoleto'    => $this->calculaSaldoAAdiantarParaOperacao('PAGAMENTO_BOLETO', $_valorTotalOperacao),
            'taxaOpTransfInterna' => $this->calculaTaxaAdiantamentoParaOperacao('TRANSFERENCIA_INTERNA', $_valorTotalOperacao),
            'taxaOpTransfExterna' => $this->calculaTaxaAdiantamentoParaOperacao('TRANSFERENCIA_EXTERNA', $_valorTotalOperacao),
            'taxaOpBoleto'        => $this->calculaTaxaAdiantamentoParaOperacao('PAGAMENTO_BOLETO', $_valorTotalOperacao),
        );
        return $result;
    }

    private function getFormaNegociadaPagamentoBoleto()
    {
        return $this->clientModel->paymentspercentage;
    }
    private function getTaxaNegociadaPagamentoBoleto()
    {
        return $this->clientModel->payments;
    }

    //Fabio - Novembro 2022
    //Tipo e valor de taxa cobrada na transferencia
    private function getFormaNegociadaTransferencia()
    {
        return $this->clientModel->transferspercentage;
    }
    private function getTaxaNegociadaTransferencia()
    {
        return $this->clientModel->transfers;
    }

    private function getFormaNegociadaTransferenciaExterna()
    {
        return $this->clientModel->transferspercentage;
    }
    private function getTaxaNegociadaTransferenciaExterna()
    {
        return $this->clientModel->transfers;
    }
    private function getFormaNegociadaTransferenciaInterna()
    {
        return $this->clientModel->paymentspercentage;
    }
    private function getTaxaNegociadaTransferenciaInterna()
    {
        return 0;
    }

    private function getFormaNegociadaAdiantamento()
    {
        return $this->clientModel->balanceusagepercentage;
    }
    private function getTaxaNegociadaAdiantamento()
    {
        return $this->clientModel->balanceusage;
    }

    //Função que calcula o limite total da operação
    function calculaLimiteDaOperacao($_operacao, ?bool $_usarSaldoCompensar)
    {

        $nomeOperacao = strtoupper($_operacao);

        switch ($nomeOperacao) {

                //Utilizado em produção para pagamento de boletos
                //O calculo foi validado na ultima atualização do pagamento de boleto (outubro de 2022)
            case 'PAGAMENTO_BOLETO':
                return $this->operationLimitCalculator(
                    $_usarSaldoCompensar,
                    $this->getFormaNegociadaPagamentoBoleto(),
                    $this->getTaxaNegociadaPagamentoBoleto()
                );

                //Implementado em Nov 2022 - Fábio
            case 'TRANSFERENCIA':
                return $this->operationLimitCalculator(
                    $_usarSaldoCompensar,
                    $this->getFormaNegociadaTransferencia(),
                    $this->getTaxaNegociadaTransferencia()
                );

                //Não utilizada em operação
                //Utilizada apenas no TesteController.php
            case 'TRANSFERENCIA_EXTERNA':
                return $this->operationLimitCalculator(
                    $_usarSaldoCompensar,
                    $this->getFormaNegociadaTransferenciaExterna(),
                    $this->getTaxaNegociadaTransferenciaExterna()
                );

                //Não utilizada em operação
                //Utilizada apenas no TesteController.php
            case 'TRANSFERENCIA_INTERNA':
                return $this->operationLimitCalculator(
                    $_usarSaldoCompensar,
                    $this->getFormaNegociadaTransferenciaInterna(),
                    $this->getTaxaNegociadaTransferenciaInterna()
                );

            case "DX_AUTOMATICO":
                return $this->operationLimitCalculator(
                    $_usarSaldoCompensar,
                    $this->getFormaNegociadaTransferenciaExterna(),
                    $this->getTaxaNegociadaTransferenciaExterna(),
                    FALSE
                );

            case "D0_AUTOMATICO":
                return $this->operationLimitCalculator(
                    TRUE,
                    $this->getFormaNegociadaTransferenciaExterna(),
                    $this->getTaxaNegociadaTransferenciaExterna(),
                    TRUE
                );

            case "D1_AUTOMATICO":
                return $this->operationLimitCalculator(
                    FALSE,
                    $this->getFormaNegociadaTransferenciaExterna(),
                    $this->getTaxaNegociadaTransferenciaExterna(),
                    TRUE
                );

            default:
                return 0;
        }
    }


    function calculaSaldoAAdiantarParaOperacao($_operacao, $_valorTotalDaOperacao)
    {
        $nomeOperacao = strtoupper($_operacao);
        switch ($nomeOperacao) {
            case 'PAGAMENTO_BOLETO':
                return $this->calculaSaldoTotalAAdiantar(
                    $_valorTotalDaOperacao,
                    $this->getFormaNegociadaPagamentoBoleto(),
                    $this->getTaxaNegociadaPagamentoBoleto()
                );

            case 'TRANSFERENCIA_EXTERNA':
                return $this->calculaSaldoTotalAAdiantar(
                    $_valorTotalDaOperacao,
                    $this->getFormaNegociadaTransferenciaExterna(),
                    $this->getTaxaNegociadaTransferenciaExterna()
                );

            case 'TRANSFERENCIA_INTERNA':
                return $this->calculaSaldoTotalAAdiantar(
                    $_valorTotalDaOperacao,
                    $this->getFormaNegociadaTransferenciaInterna(),
                    $this->getTaxaNegociadaTransferenciaInterna()
                );

            case 'DX_AUTOMATICO':
                return $this->calculaSaldoTotalAAdiantar(
                    $_valorTotalDaOperacao,
                    $this->getFormaNegociadaTransferenciaExterna(),
                    $this->getTaxaNegociadaTransferenciaExterna()
                );

            default:
                return 0;
        }
    }

    function calculaTaxaAdiantamentoParaOperacao($_operacao, $_valorTotalDaOperacao)
    {
        $nomeOperacao = strtoupper($_operacao);
        switch ($nomeOperacao) {
            case 'PAGAMENTO_BOLETO':
                return $this->calculaValorTaxaAdiantamento(
                    $_valorTotalDaOperacao,
                    $this->getFormaNegociadaPagamentoBoleto(),
                    $this->getTaxaNegociadaPagamentoBoleto()
                );

            case 'TRANSFERENCIA_EXTERNA':
                return $this->calculaValorTaxaAdiantamento(
                    $_valorTotalDaOperacao,
                    $this->getFormaNegociadaTransferenciaExterna(),
                    $this->getTaxaNegociadaTransferenciaExterna()
                );

            case 'TRANSFERENCIA_INTERNA':
                return $this->calculaValorTaxaAdiantamento(
                    $_valorTotalDaOperacao,
                    $this->getFormaNegociadaTransferenciaInterna(),
                    $this->getTaxaNegociadaTransferenciaInterna()
                );

            case 'DX_AUTOMATICO':
                return $this->calculaValorTaxaAdiantamento(
                    $_valorTotalDaOperacao,
                    $this->getFormaNegociadaTransferenciaExterna(),
                    $this->getTaxaNegociadaTransferenciaExterna(),
                    FALSE
                );

            case 'D0_AUTOMATICO':
                return $this->calculaValorTaxaAdiantamento(
                    $_valorTotalDaOperacao,
                    $this->getFormaNegociadaTransferenciaExterna(),
                    $this->getTaxaNegociadaTransferenciaExterna(),
                    TRUE
                );

            case 'D1_AUTOMATICO':
                return $this->calculaValorTaxaAdiantamento(
                    $_valorTotalDaOperacao,
                    $this->getFormaNegociadaTransferenciaExterna(),
                    $this->getTaxaNegociadaTransferenciaExterna(),
                    TRUE
                );

            default:
                return 0;
        }
    }

    function calculaTaxaDaOperacao($_operacao, $_valorTotalDaOperacao)
    {
        $nomeOperacao = strtoupper($_operacao);
        switch ($nomeOperacao) {
            case 'PAGAMENTO_BOLETO':
                return $this->calculaValorTaxaOperacao(
                    $_valorTotalDaOperacao,
                    $this->getFormaNegociadaPagamentoBoleto(),
                    $this->getTaxaNegociadaPagamentoBoleto()
                );

            case 'TRANSFERENCIA_EXTERNA':
                return $this->calculaValorTaxaOperacao(
                    $_valorTotalDaOperacao,
                    $this->getFormaNegociadaTransferenciaExterna(),
                    $this->getTaxaNegociadaTransferenciaExterna()
                );

            case 'TRANSFERENCIA_INTERNA':
                return $this->calculaValorTaxaOperacao(
                    $_valorTotalDaOperacao,
                    $this->getFormaNegociadaTransferenciaInterna(),
                    $this->getTaxaNegociadaTransferenciaInterna()
                );

            case 'DX_AUTOMATICO':
                return $this->calculaValorTaxaOperacao(
                    $_valorTotalDaOperacao,
                    $this->getFormaNegociadaTransferenciaExterna(),
                    $this->getTaxaNegociadaTransferenciaExterna()
                );

            case 'D0_AUTOMATICO':
                return $this->calculaValorTaxaOperacao(
                    $_valorTotalDaOperacao,
                    $this->getFormaNegociadaTransferenciaExterna(),
                    $this->getTaxaNegociadaTransferenciaExterna()
                );

            case 'D1_AUTOMATICO':
                return $this->calculaValorTaxaOperacao(
                    $_valorTotalDaOperacao,
                    $this->getFormaNegociadaTransferenciaExterna(),
                    $this->getTaxaNegociadaTransferenciaExterna()
                );

            case 'ADIANTAMENTO':
                return $this->calculaValorTaxaOperacao(
                    $_valorTotalDaOperacao,
                    $this->getFormaNegociadaAdiantamento(),
                    $this->getTaxaNegociadaAdiantamento()
                );

            default:
                return 0;
        }
    }

    // Calcula o valor total que deverá ser enviado para transferencia no Fitbank
    private function calculaSaldoTotalAAdiantar($_valorTotalDaOperacao, $_calcularTaxaPercentual, $_valorTaxaNegociada)
    {

        $taxaOperacao = $this->calculaValorTaxaOperacao($_valorTotalDaOperacao, $_calcularTaxaPercentual, $_valorTaxaNegociada);

        $valorDiferenca = $_valorTotalDaOperacao + $taxaOperacao - $this->getSaldoRealPaged();
        if ($valorDiferenca <= 0) {
            return 0;
        }

        $taxaAdiantamento = $this->calculaValorTaxaAdiantamento($_valorTotalDaOperacao, $_calcularTaxaPercentual, $_valorTaxaNegociada);

        $valorAAdiantar = $valorDiferenca + $taxaAdiantamento;

        return $valorAAdiantar;
    }

    //Modificado em Novembro 2022 por Fábio
    //Calcula o limite total da operação
    private function operationLimitCalculator($_usarSaldoCompensar, $_calcularTaxaPercentual, $_valorTaxaNegociada, $_usarSaldoReal = TRUE)
    {

        // Calcula limite da operação de acordo com opção de adiantamento

        if ($_usarSaldoReal) {
            $saldoCliente = $this->getSaldoRealPaged();
        } else {
            $saldoCliente = 0.0;
        }

        $saldoCompensar = $this->getSaldoCompensar();

        $saldoTotal = $saldoCliente + $saldoCompensar;

        //Calcula taxa da operação
        //Pode ser percentual ou valor fixo
        //Atenção: essa é a taxa máxima que poderá ser cobrada (deixando uma margem de segurança). Pois leva em conta a utilização
        //do saldo total do cliente (saldo + saldo a compensar)
        $taxaOperacao = $this->calculaValorTaxaOperacao($saldoTotal, $_calcularTaxaPercentual, $_valorTaxaNegociada);

        //Valor total provisionado do cliente
        $valorTotalProvisionado = calculaValorTotalProvisionado($this->taxnumber);

        //SEM a utilização do SALDO A COMPENSAR
        $limiteOperacao = $saldoCliente - $taxaOperacao - $valorTotalProvisionado;

        //COM a utilização do SALDO A COMPENSAR
        if ($_usarSaldoCompensar) {

            $taxaAdiantamento = $this->calculaValorTaxaAdiantamento($saldoTotal - $taxaOperacao, $_calcularTaxaPercentual, $_valorTaxaNegociada, $_usarSaldoReal);

            $limiteOperacao = $saldoTotal - $taxaOperacao - $taxaAdiantamento - $valorTotalProvisionado;
        }

        if ($limiteOperacao < 0) {
            $limiteOperacao = 0;
        }

        return (floor($limiteOperacao * 100) / 100);
    }

    /**
     * Calcular o valor do cashback que será cobrado do cliente
     */
    private function calculaValorTaxaAdiantamento($_valorTotalDaOperacao, $_calcularTaxaPercentualOperacao, $_valorTaxaNegociadaOperacao, $_usarSaldoReal = TRUE)
    {

        if ($_usarSaldoReal) {
            $saldoReal = $this->getSaldoRealPaged();
        } else {
            $saldoReal = 0.0;
        }

        $taxaOperacao = $this->calculaValorTaxaOperacao($_valorTotalDaOperacao, $_calcularTaxaPercentualOperacao, $_valorTaxaNegociadaOperacao);

        $valorDiferenca = $_valorTotalDaOperacao + $taxaOperacao - $saldoReal;
        if ($valorDiferenca <= 0) {
            return 0;
        }

        //Calculo para Operacao = Adiantamento
        $calcularTaxaPercentualAdiantamento = $this->clientModel->balanceusagepercentage;
        $valorTaxaNegociadaAdiantamento = $this->clientModel->balanceusage;
        return $this->calculaValorTaxaOperacao($valorDiferenca, $calcularTaxaPercentualAdiantamento, $valorTaxaNegociadaAdiantamento);
    }

    private function calculaValorTaxaOperacao($_valorTotalDaOperacao, $_calcularTaxaPercentual, $_valorTaxaNegociada)
    {
        $taxaOperacao = $_valorTaxaNegociada;

        if ($_calcularTaxaPercentual == 1) {
            $taxaOperacao = ($_valorTaxaNegociada * $_valorTotalDaOperacao) / 100;
        }
        return $taxaOperacao;
    }

    private function identifyHierarchy()
    {
        $client = $this->clientModel;

        if ($client->taxnumber == $this->taxnumber) {
            return 'M';
        } else {
            $this->multisafeModel = Multisafe::where('taxnumber', '=', $this->taxnumber)
                ->where("active", "=", "1")
                ->first();
            return 'F';
        }
    }

    private function getModelClient()
    {
        $client =  Client::select('clients.*')
            ->where('id', '=', $this->clientId)
            ->first();

        // se o cliente não possuir taxas customizadas, deve usar as padrão SSG
        if ($client->customrates == '0') {
            $sps = getSPS();

            $client->balanceusagepercentage = $sps->clientbalanceusagepercentage;
            $client->balanceusage = $sps->clientbalanceusage;

            $client->transferspercentage = $sps->clienttransferspercentage;
            $client->transfers = $sps->clienttransfers;

            $client->billets = $sps->clientbillets;

            $client->paymentspercentage = $sps->clientpaymentspercentage;
            $client->payments = $sps->clientpayments;
        }
        return $client;
    }

    private function resolveClientId()
    {
        // $clientId = Auth::user()->client_id;

        // if ( !Auth::user() ) {
        //     $clientId = 0;
        // } else {
        //     if ( Auth::user()->role_id <= 1) {
        //         $clientId = 0;
        //     } else {
        //         $clientId = Auth::user()->client_id;
        //     }
        // }

        // if (0 == $clientId) {
        $client = Client::where('taxnumber', '=', $this->taxnumber)
            ->where('active', '=', 1)
            ->first();

        if (!$client) {
            $multi = Multisafe::where('taxnumber', '=', $this->taxnumber)
                ->where('active', '=', 1)
                ->first();
            $clientId = $multi->client_id;
        } else {
            $clientId = $client->id;
        }
        // }

        return $clientId;
    }

    private function onlyNumbersTaxnumber($_taxnumber)
    {
        // deixar apenas numeros
        return preg_replace('/[^0-9]/is', '', $_taxnumber);
    }

    // SETTERS & GETTERS
    public function __set($_name, $_value)
    {
        switch ($_name) { //this is kind of silly example, bt shows the idea
            case 'saldoCompensar':
                return $this->setSaldoCompensar($_value);
            case 'saldoReal':
                return $this->setSaldoReal($_value);
            case 'saldoRealPaged':
                return $this->setSaldoReal($_value);
            case 'nickname':
                return $this->setNickname($_value);
            default:
                if (property_exists($this, $_name)) {
                    $this->$_name = $_value;
                }
        }
    }
    public function __get($_name)
    {
        switch ($_name) {
            case 'saldoCompensar':
                return $this->getSaldoCompensar();
            case 'saldoReal':
                return $this->getSaldoReal();
            case 'saldoRealPaged':
                return $this->getSaldoRealPaged();
            case 'nickname':
                return $this->getNickname();
            default:
                if (property_exists($this, $_name)) {
                    return $this->$_name;
                }
        }
    }

    private function setSaldoCompensar($_value)
    {
        //$this->saldoCompensar = $_saldoCompensar;
    }

    public function getSaldoCompensar()
    {
        if ($this->hierarquia == 'M') {
            return $this->clientModel->availablelimit;
        } else {
            return $this->multisafeModel->availablelimit;
        }
    }

    private function setSaldoReal($_value)
    {
        //$this->saldoCompensar = $_saldoCompensar;
    }
    private function getSaldoReal()
    {
        return saldoCliente($this->taxnumber);
    }
    private function getSaldoRealPaged()
    {
        return saldoCliente($this->taxnumber);
    }
    private function setNickname($_value)
    {
        //$this->saldoCompensar = $_saldoCompensar;
    }
    private function getNickname()
    {
        if ($this->hierarquia == 'M') {
            return $this->clientModel->nickname;
        } else {
            return $this->multisafeModel->nickname;
        }
    }
}

/*
    function calculaTaxaDaOperacaoOLD($_operacao, $_valorTotalDaOperacao) {
        $nomeOperacao = strtoupper($_operacao);
        switch($nomeOperacao) {
            case 'PAGAMENTO_BOLETO':
                return $this->getOperationRate_SPSPagamentoBoleto($_valorTotalDaOperacao);
            case 'TRANSFERENCIA_EXTERNA':
                return $this->getOperationRate_SPSTransferencia($_valorTotalDaOperacao);
            case 'TRANSFERENCIA_INTERNA':
                return $this->getOperationRate_SPSTransferenciaInterna($_valorTotalDaOperacao);
            default:
                return 0;
        }
    }

    private function getOperationRate_SPSPagamentoBoleto($_valorTotalDaOperacao) {
        return $this->calculaValorTaxaOperacao(
                            $_valorTotalDaOperacao,
                            $this->getFormaNegociadaPagamentoBoleto(),
                            $this->getTaxaNegociadaPagamentoBoleto());
    }

    private function getOperationRate_SPSTransferencia($_valorTotalDaOperacao) {
        return $this->calculaValorTaxaOperacao(
                            $_valorTotalDaOperacao,
                            $this->getFormaNegociadaTransferenciaExterna(),
                            $this->getTaxaNegociadaTransferenciaExterna());
    }

    private function getOperationRate_SPSTransferenciaInterna($_valorTotalDaOperacao) {
        return $this->calculaValorTaxaOperacao(
                            $_valorTotalDaOperacao,
                            $this->getFormaNegociadaTransferenciaInterna(),
                            $this->getTaxaNegociadaTransferenciaInterna());
    }

    function calculaTaxaAdiantamentoParaOperacaoOLD($_operacao, $_valorTotalDaOperacao) {
        $nomeOperacao = strtoupper($_operacao);
        switch($nomeOperacao) {
            case 'PAGAMENTO_BOLETO':
                return $this->getAdvanceRate_SPSPagamentoBoleto($_valorTotalDaOperacao);
            case 'TRANSFERENCIA_EXTERNA':
                return $this->getAdvanceRate_SPSTransferencia($_valorTotalDaOperacao);
            case 'TRANSFERENCIA_INTERNA':
                return $this->getAdvanceRate_SPSTransferenciaInterna($_valorTotalDaOperacao);
            default:
                return 0;
        }
    }

    private function getAdvanceRate_SPSPagamentoBoleto($_valorTotalDaOperacao) {
        return $this->calculaValorTaxaAdiantamento(
                            $_valorTotalDaOperacao,
                            $this->getFormaNegociadaPagamentoBoleto(),
                            $this->getTaxaNegociadaPagamentoBoleto());
    }

    private function getAdvanceRate_SPSTransferencia($_valorTotalDaOperacao) {
        return $this->calculaValorTaxaAdiantamento(
                            $_valorTotalDaOperacao,
                            $this->getFormaNegociadaTransferenciaExterna(),
                            $this->getTaxaNegociadaTransferenciaExterna());
    }

    private function getAdvanceRate_SPSTransferenciaInterna($_valorTotalDaOperacao) {
        return $this->calculaValorTaxaAdiantamento(
                            $_valorTotalDaOperacao,
                            $this->getFormaNegociadaTransferenciaInterna(),
                            $this->getTaxaNegociadaTransferenciaInterna());
    }

    function calculaLimiteDaOperacaoOLD($_operacao, $_usarSaldoCompensar) {
        $nomeOperacao = strtoupper($_operacao);
        switch($nomeOperacao) {
            case 'PAGAMENTO_BOLETO':
                return $this->getOperationLimit_SPSPagamentoBoleto($_usarSaldoCompensar);
            case 'TRANSFERENCIA_EXTERNA':
                return $this->getOperationLimit_SPSTransferencia($_usarSaldoCompensar);
            case 'TRANSFERENCIA_INTERNA':
                return $this->getOperationLimit_SPSTransferenciaInterna($_usarSaldoCompensar);
            default:
                return 0;
        }
    }
    private function getOperationLimit_SPSPagamentoBoleto($_usarLimiteEspecial) {
        return $this->operationLimitCalculator(
                            $_usarLimiteEspecial,
                            $this->getFormaNegociadaPagamentoBoleto(),
                            $this->getTaxaNegociadaPagamentoBoleto());
    }

    private function getOperationLimit_SPSTransferencia($_usarLimiteEspecial) {
        return $this->operationLimitCalculator(
                            $_usarLimiteEspecial,
                            $this->getFormaNegociadaTransferenciaExterna(),
                            $this->getTaxaNegociadaTransferenciaExterna());
    }

    private function getOperationLimit_SPSTransferenciaInterna($_usarLimiteEspecial) {
        return $this->operationLimitCalculator(
                            $_usarLimiteEspecial,
                            $this->getFormaNegociadaTransferenciaInterna(),
                            $this->getTaxaNegociadaTransferenciaInterna());
    }

    function calculaSaldoAAdiantarParaOperacaoOLD($_operacao, $_valorTotalDaOperacao) {
        $nomeOperacao = strtoupper($_operacao);
        switch($nomeOperacao) {
            case 'PAGAMENTO_BOLETO':
                return $this->getAdvanceValue_SPSPagamentoBoleto($_valorTotalDaOperacao);
            case 'TRANSFERENCIA_EXTERNA':
                return $this->getAdvanceValue_SPSTransferencia($_valorTotalDaOperacao);
            case 'TRANSFERENCIA_INTERNA':
                return $this->getAdvanceValue_SPSTransferenciaInterna($_valorTotalDaOperacao);
            default:
                return 0;
        }
    }

    private function getAdvanceValue_SPSPagamentoBoleto($_valorTotalDaOperacao) {
        return $this->calculaSaldoTotalAAdiantar(
                            $_valorTotalDaOperacao,
                            $this->getFormaNegociadaPagamentoBoleto(),
                            $this->getTaxaNegociadaPagamentoBoleto());
    }

    private function getAdvanceValue_SPSTransferencia($_valorTotalDaOperacao) {
        return $this->calculaSaldoTotalAAdiantar(
                            $_valorTotalDaOperacao,
                            $this->getFormaNegociadaTransferenciaExterna(),
                            $this->getTaxaNegociadaTransferenciaExterna());
    }

    private function getAdvanceValue_SPSTransferenciaInterna($_valorTotalDaOperacao) {
        return $this->calculaSaldoTotalAAdiantar(
                            $_valorTotalDaOperacao,
                            $this->getFormaNegociadaTransferenciaInterna(),
                            $this->getTaxaNegociadaTransferenciaInterna());
    }

    // FUNÇÃO ORIGINALMENTE CRIADA COM ROBSON
    // armazenada aqui em 06/02/2022 para aguardar validação da r_1.5.1.01
    private function getOperationLimit_SPSTransferencia_TST1($_usarLimiteEspecial) {
        $usarSaldoCompensar = false;
        $limiteoperacao = 0;

        if ($_usarLimiteEspecial == 1) {
            $usarSaldoCompensar = true;
        }

        // Calcula limite da operação de acordo com opção de adiantamento
        $saldoCliente = $this->getSaldoReal();
        $saldoCompensar = $this->getSaldoCompensar();
        $client = $this->clientModel;

        if ($client->transferspercentage == 1) {
            $taxaTransferencia = ($client->transfers * ($saldoCliente + $saldoCompensar) ) /100;
        } else {
            $taxaTransferencia = $client->transfers;
        }
        if ($client->balanceusagepercentage == 1) {
            $taxaAdiantamento = ($client->balanceusage * $saldoCompensar ) / 100;
        } else {
            $taxaAdiantamento = $client->balanceusage;
        }

        // - verifica opção de adiantamento para calcular limite
        if ($usarSaldoCompensar) {
            $limiteoperacao = $saldoCliente + $saldoCompensar
                            - $taxaTransferencia - $taxaAdiantamento;
        } else {
            $limiteoperacao = $saldoCliente;
        }
        if ($limiteoperacao < 0) {
            $limiteoperacao = 0;
        }

        return $limiteoperacao;
    }
*/
