<?php

namespace App\Console\Commands;

use App\Autotransferreceiver;
use App\Autotransferschedule;
use App\Classes\ClientSPS\ClientSpsService;
use Carbon\Carbon;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Artisan;

class ExecutarDxAutomatico extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'command:executardxautomatico';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Executa todos os D+x que estão agendados';

    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        Artisan::call('command:agendardxautomatico');

        $diaHojeCompleto = date('Y-m-d');
        if (!ehDiaUtil($diaHojeCompleto)) return false;

        // 1) Busca todos os D+0 que estão agendados
        // TIP
        // Para testar esta funcionalidade, descomente o whereDate() e comente o whereRaw()
        $todosAgendados = Autotransferschedule::where('operationstatus', '=', 'AGENDADO')
            // ->whereDate('scheduleddatetime', '=', $diaHojeCompleto)
            // ->whereRaw('scheduleddatetime LIKE "' . date('Y-m-d H:') . '%:%"')
            ->whereBetween('scheduleddatetime', [date('Y-m-d 00'), date('Y-m-d H')])
            ->get();

        foreach ($todosAgendados as $agendamento) {

            if (taxnumberEstaDuplicado($agendamento->fromtaxnumber)) {
                $agendamento->operationstatus = "ERRO_INCONSISTENTE";
                $agendamento->operationresponse = "CPF/CNPJ ativo como mais de um cliente. [EXECUÇÃO: " . datetime2br(date('Y-m-d H:i:s')) . "]";
                $agendamento->update();

                continue;
            }


            $objCliente = $this->checarSeClienteEstaAtivo($agendamento->fromtaxnumber);
            if (!$objCliente) {
                $agendamento->operationstatus = "ERRO_DESATIVADO";
                $agendamento->operationresponse = "CPF/CNPJ desativado. [EXECUÇÃO: " . datetime2br(date('Y-m-d H:i:s')) . "]";
                $agendamento->update();

                continue;
            }


            $clientSpsService = new ClientSpsService($objCliente->taxnumber);

            if ($agendamento->operationtype == "D+0") {
                $this->executarD0Automatico($objCliente, $agendamento, $clientSpsService);
            } elseif ($agendamento->operationtype == "D+1") {
                $this->executarD1Automatico($objCliente, $agendamento, $clientSpsService);
            }
        }
    }

    /**
     * Busca um cliente ativo pelo seu taxnumber.
     * @param string $taxnumber Taxnumber do cliente
     * @return object|boolean Retorna o objeto do cliente caso ele esteja ativo, senão, retorna FALSE
     */
    public function checarSeClienteEstaAtivo($taxnumber)
    {
        $objCliente = buscarClienteOuMultisafePeloTaxnumber($taxnumber);
        return $objCliente;
    }

    /**
     * Executa todos os agendamentos D+0, que são as operações de TED utilizando o saldo real + o saldo a compensar
     * @param object $objCliente Objeto do cliente
     * @param object $agendamento Objeto do agendamento em autotransferschedule
     */
    public function executarD0Automatico($objCliente, $agendamento, $clientSpsService) {

        //  RETORNA O "availablelimit" DO CLIENT OU MULTISAFE
        $availablelimitInicial = getAvailableLimitMultiCnpj($objCliente->taxnumber);

        // TRANSCRIÇÃO DE NOME 
        $saldoCompensar = $availablelimitInicial;

        // FAZ A CHAMADA A API DO FIT BANK "GetAccountEntryPaged" E RETORNA O "Balance"
        $saldoReal = $clientSpsService->__get('saldoRealPaged');

        $adiantamentoStatus = null;
        $adiantamentoResponse = null;
        $notificarErro = null;


        if (($saldoCompensar + $saldoReal) <= 0) {
            $agendamento->operationstatus = "CONCLUIDO";
            $agendamento->availablelimit = $availablelimitInicial;
            $agendamento->operationvalue = $saldoCompensar;
            $agendamento->operationresponse = "CPF/CNPJ não possuí saldo. [EXECUÇÃO: " . datetime2br(date('Y-m-d H:i:s')) . "]";
            $agendamento->update();
            //TODO: LANÇAR INFORMAÇÃO NO LOG
            return false;
        }
        
        // calculaLimiteDaOperacao RECEBE O QUE DEVE SER EXECUTADO E SE DEVE SER UTILIZAR O SALDO A COMPENSAR OU NÃO 
        // QUAL O MOTIVO DE PASSAR NULL EM UM CAMPO QUE DEVERIA SER BOOL
        $limiteOperacao = $clientSpsService->calculaLimiteDaOperacao("D0_AUTOMATICO", null);
        $valorOperacao = $limiteOperacao;

        $taxaD0 = $clientSpsService->calculaTaxaDaOperacao("D0_AUTOMATICO", $limiteOperacao);

        $taxaAdiantamento = $clientSpsService->calculaTaxaAdiantamentoParaOperacao("D0_AUTOMATICO", $limiteOperacao);

        $valorAAdiantar = $limiteOperacao + $taxaD0 - $saldoReal;
        $valorAdiantado = 0;

        if ($valorAAdiantar > 0) {

            $adiantouSaldoACompensar = transfereSaldoDisponivelCliente(
                $agendamento->fromtaxnumber,
                $valorAAdiantar,
                'Adiantamento ' . $agendamento->operationtype . ' automático',
                $taxaAdiantamento,
                0
            );


            if ($adiantouSaldoACompensar) {

                logAvailableLimitUsed(
                    1,
                    getClientByTaxnumber($agendamento->fromtaxnumber)->id,
                    $agendamento->fromtaxnumber,
                    $saldoCompensar,
                    $valorAAdiantar,
                    'ADIANTAMENTO',
                    'SSG_' . $agendamento->operationtype . '_AUTOTRANSFERSCHEDULE-' . $agendamento->id,
                    false,
                    false,
                    $taxaAdiantamento,
                    date('Y-m-d', strtotime("+1 day")) . ' 09:00:00'
                );

                $adiantamentoResponse .= "Adiantamento executado com sucesso.";
                $valorAdiantado = $valorAAdiantar;

            } else {
                // TODO
                // Se não for executado o adiantamento
                // é possível continuar com a operação apenas com o saldo real ou realmente é para parar e retornar erro?

                $adiantamentoStatus = "ADIANTAMENTO-FALHOU";
                $adiantamentoResponse .= "Erro ao adiantar o saldo a compensar.";
                $valorOperacao = $limiteOperacao - $valorAAdiantar;

            }

        }

        // Transferência para destinatário cadastrado
        $objDestinatario = Autotransferreceiver::where("operationtype", "=", $agendamento->operationtype)
            ->where("totaxnumber", "=", $agendamento->totaxnumber)
            ->where("fromtaxnumber", "=", $agendamento->fromtaxnumber)
            ->first();

        $attempts = 5;
        $delay = 5;

        for ($i = 0; $i < $attempts; $i++) {
            $saldoReal = $clientSpsService->__get('saldoRealPaged');

            if ($saldoReal < $valorOperacao) {
                sleep($delay);
            } else {
                break;
            }
        }

        $dataAgendamento = Carbon::parse($agendamento->scheduleddatetime);

        /*Verificar se o horario do agendamento D+0 é 19:00 ou maior
        * Se for sextafeira, a transferência será agendada para o próxima segunda-feira
        * Se for, a transferência será agendada para o próximo dia útil - o Registro do envio da transferencia pode ser checado na transferLogs
        */
        if ($dataAgendamento->format('H') >= 19){

            if ($dataAgendamento->format('N') == 5){
                $dataAgendamento = date('Y-m-d 10:00:00', strtotime($agendamento->scheduleddatetime . ' +3 day'));
            } else {
                $dataAgendamento = date('Y-m-d 10:00:00', strtotime($agendamento->scheduleddatetime . ' +1 day'));
            }
            $arrayTED = $this->executarTEDComData($clientSpsService, $agendamento, $valorOperacao, $objDestinatario, $valorAdiantado, $dataAgendamento );
        } else {
            $arrayTED = $this->executarTED($clientSpsService, $agendamento, $valorOperacao, $objDestinatario, $valorAdiantado);
        }



        $agendamento->operationstatus = $arrayTED["status"] . ( ($adiantamentoStatus) ? "_$adiantamentoStatus" : null );
        $agendamento->operationresponse = $adiantamentoResponse . " " . $arrayTED["response"] . "[EXECUÇÃO: " . datetime2br(date('Y-m-d H:i:s')) ."]";
        $agendamento->documentnumber = $arrayTED["documentNumber"];
        $agendamento->availablelimit = $availablelimitInicial;
        $agendamento->operationvalue = $valorOperacao;

        $agendamento->update();
    }

    /**
     * Executa todos os agendamentos D+1, que são as operações de TED utilizando apenas o saldo real
     * @param object $objCliente Objeto do cliente
     * @param object $agendamento Objeto do agendamento em autotransferschedule
     */
    public function executarD1Automatico($objCliente, $agendamento, ClientSpsService $clientSpsService)
    {
        //verificar se o valor da operação será total ou já descontado (forma q está sendo utilizada)
        $valorOperacao = $clientSpsService->calculaLimiteDaOperacao("D1_AUTOMATICO", null);//$clientSpsService->__get
        $availablelimitInicial = getAvailableLimitMultiCnpj($objCliente->taxnumber);

        $objDestinatario = Autotransferreceiver::where("operationtype", "=", $agendamento->operationtype)
            ->where("totaxnumber", "=", $agendamento->totaxnumber)
            ->where("fromtaxnumber", "=", $agendamento->fromtaxnumber)
            ->first();

        $arrayTED = $this->executarTED($clientSpsService, $agendamento, $valorOperacao, $objDestinatario, 0);

        $agendamento->operationstatus = $arrayTED["status"];
        $agendamento->operationresponse = $arrayTED["response"] . "[EXECUÇÃO: " . datetime2br(date('Y-m-d H:i:s')) ."]";
        $agendamento->documentnumber = $arrayTED["documentNumber"];
        $agendamento->availablelimit = $availablelimitInicial;
        $agendamento->operationvalue = $valorOperacao;

        $agendamento->update();
    }

    /**
     * Executa o TED do agendamento
     * @param object $clientSpsService
     * @param object $agendamento
     * @param float $valorOperacao
     * @param object $objDestiantario
     * @param float $valorAdiantado Valor que foi adiantado do saldo a compensar
     * @return array $arrayRetorno status, response e documentnumber
     */
    public function executarTED($clientSpsService, $agendamento, $valorOperacao, $objDestinatario, $valorAdiantado)
    {
        $taxaTransferencia = $clientSpsService->calculaTaxaDaOperacao(
            "TRANSFERENCIA_EXTERNA",
            $valorOperacao
        );

        $usouSaldoCompensar = ($valorAdiantado > 0) ? TRUE : FALSE;

        $transferiuParaDestinatario = transferirParaDestinatarioExterno(
            $agendamento->fromtaxnumber,
            $agendamento->totaxnumber,
            // buscarClienteOuMultisafePeloTaxnumber($agendamento->totaxnumber)->nickname,
            $objDestinatario->nickname,
            $agendamento->bank_id,
            $agendamento->bankbranch,
            $agendamento->bankaccount,
            $agendamento->bankaccountdigit,
            $agendamento->accounttype,
            $valorOperacao,
            $taxaTransferencia,
            $objDestinatario->identifier,
            $usouSaldoCompensar,
            $valorAdiantado
        );

        if (!$transferiuParaDestinatario["Success"]) {
            $arrayRetorno = [
                "success" => FALSE,
                "status" => $transferiuParaDestinatario["ErrorCode"],
                "response" => $transferiuParaDestinatario["Message"],
                "documentNumber" => "0"
            ];
        } else {
            $arrayRetorno = [
                "success" => TRUE,
                "status" => "CONCLUIDO",
                "response" => $transferiuParaDestinatario["Message"],
                "documentNumber" => $transferiuParaDestinatario["DocumentNumber"]
            ];
        }

        return $arrayRetorno;
    }

    /**
     * Executa o TED do agendamento
     * @param object $clientSpsService
     * @param object $agendamento
     * @param float $valorOperacao
     * @param object $objDestiantario
     * @param float $valorAdiantado Valor que foi adiantado do saldo a compensar
     * @return array $arrayRetorno status, response e documentnumber
     */
    public function executarTEDComData($clientSpsService, $agendamento, $valorOperacao, $objDestinatario, $valorAdiantado,$dataAgendamento)
    {
        $taxaTransferencia = $clientSpsService->calculaTaxaDaOperacao(
            "TRANSFERENCIA_EXTERNA",
            $valorOperacao
        );

        $usouSaldoCompensar = ($valorAdiantado > 0) ? TRUE : FALSE;

        $transferiuParaDestinatario = transferirParaDestinatarioExternoComData(
            $agendamento->fromtaxnumber,
            $agendamento->totaxnumber,
            // buscarClienteOuMultisafePeloTaxnumber($agendamento->totaxnumber)->nickname,
            $objDestinatario->nickname,
            $agendamento->bank_id,
            $agendamento->bankbranch,
            $agendamento->bankaccount,
            $agendamento->bankaccountdigit,
            $agendamento->accounttype,
            $valorOperacao,
            $taxaTransferencia,
            $objDestinatario->identifier,
            $usouSaldoCompensar,
            $valorAdiantado,
            $dataAgendamento
        );

        if (!$transferiuParaDestinatario["Success"]) {
            $arrayRetorno = [
                "success" => FALSE,
                "status" => $transferiuParaDestinatario["ErrorCode"],
                "response" => $transferiuParaDestinatario["Message"],
                "documentNumber" => "0"
            ];
        } else {
            $arrayRetorno = [
                "success" => TRUE,
                "status" => "CONCLUIDO",
                "response" => $transferiuParaDestinatario["Message"],
                "documentNumber" => $transferiuParaDestinatario["DocumentNumber"]
            ];
        }

        return $arrayRetorno;
    }
}
