<?php

namespace App\Http\Controllers;

use App\Autotransferreceiver;
use App\Bank;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Session;
use Illuminate\Http\Request;

use App\Client;
use App\User;
use App\Role;
use App\Paymentlog;
use App\Transferlog;
use App\Clientcontact;
use App\Billet;
use App\Billeterror;
use App\Darfpayment;
use App\Darjpayment;
use App\Fgtspayment;
use App\Gpspayment;
use App\Garepayment;
use App\Barcodepayment;
use App\Usedavailablelimit;
use App\CollectsMap;

use App\Classes\ClientSPS\ClientSpsService;

use App\Multisafe;
use App\Transfermanager;
use Illuminate\Support\Facades\Validator;
use Normalizer;
use PHPUnit\Exception;

class ClientWebController extends Controller
{
	public function __construct()
	{
		$this->middleware('auth');
	}


	public function extrato(Request $request)
	{
		$client = Client::select('clients.*')->where('id', '=', Auth::user()->client_id)->first();

		$spsInfo = getSPS();

		if (isset($request) && ($request->datade != '') && ($request->dataate != '') && (dataSlashReverse($request->datade) <= dataSlashReverse($request->dataate))) {
			$datade = date(dataSlashReverse($request->datade), strtotime('-15 days'));
			$dataate = date(dataSlashReverse($request->dataate));
		} else {
			$datade = date('Y/m/d', strtotime('-15 days'));
			$dataate = date('Y/m/d');
		}

		$client_taxnumber = Session::get('cnpj');

		$curlParams = "{ \"Method\": \"GetAccountEntry\",
                    \"PartnerId\": " . $spsInfo->partnerid . ",
                    \"BusinessUnitId\": " . $spsInfo->businessunitid . ",
                    \"TaxNumber\": \"" . $client_taxnumber . "\",
                    \"StartDate\": \"" . $datade . "\",
                    \"EndDate\": \"" . $dataate . "\"
                }";

		$extrato = sendCurl($curlParams);

		if ($extrato["Success"] == "false")
			return redirect('/home')->with('warning', 'Erro: ocorreu um problema ao processar a requisição.');

		$datafiltro = array('datade' => dataSlashReverse($datade), 'dataate' => dataSlashReverse($dataate));

		$lancamentosfuturos = $this->lancamentosFuturos($client->id);

		// $saldocompensar = getSmartsafeBalance($client->id);
		$clientMultiCnpj = getMultiCnpj($client->id);
		$clientMultiCnpjAvailableLimit = getAvailableLimitMultiCnpj($client_taxnumber);

		$saldocompensar = (empty($clientMultiCnpj)) ? $client->availablelimit : $clientMultiCnpjAvailableLimit;

		// if ($_POST['action'] == 'pdf') {
		//     print("extrato");
		// }
		// else {
		//     return view('clients.extrato',compact('extrato','datafiltro','client','lancamentosfuturos','saldocompensar'));
		// }
		return view('clients.extrato', compact('extrato', 'datafiltro', 'client', 'lancamentosfuturos', 'saldocompensar'));

	}

	public function clientes()
	{
		$clients = Client::where('active','=','1')
			->orderBy('name', 'asc')
			->get();
		// return compact('clients');
		return view('clients.index', compact('clients'));
	}

	public function clientesFiltradosPorNome(Request $request){
		$filtro = $request->all();
		$clients = Client::where('active','=','1')->orderBy('name', 'asc');

		if($filtro['tipofiltro'] == 1){
			$clients = $clients->where('name', 'like', '%' . $filtro['filtro'] . '%')->get();
			return view('clients.index', compact('clients'));
		}

        if($filtro['tipofiltro'] == 2){

            $limpo = preg_replace('/[^0-9\s]/', '', $filtro['filtro']);

			$clients = $clients->where('taxnumber', 'like', '%' . $limpo . '%')->get();
			return view('clients.index', compact('clients'));
		}

		$clients = $clients->get();
		return view('clients.index', compact('clients'));
	}


	public function verCliente(Request $request, $id)
	{
		$client = Client::findOrFail($id);
		return view('clients.vercliente', compact('client'));
	}



	public function criaCliente(Request $request)
	{
		$verificaEmail = verificaEmailExistente($request->mail, 'cliente');

		if ($verificaEmail)
			return redirect('/clientes')->with('status', 'Já existe um cliente cadastrado com o email ' . $request->email);

		if (!filter_var($request->mail, FILTER_VALIDATE_EMAIL)) {
			return redirect('/clientes')->with('status', 'O formato do email fornecido parece inválido.');
		}

		$originalNickname = $request->nickname;
		$normalized = Normalizer::normalize($originalNickname, Normalizer::NFD);
		$nickname = preg_replace('/[[`~´^]/', '', preg_replace('/[\x{0300}-\x{036F}]/u', '', $normalized));

		$request->merge([
			'nickname' => $nickname,
			'taxnumber' => somenteNumeros($request->taxnumber)
		]);

		$client = new Client($request->all());

		$client->businessunitid = '';
		$client->save();

		return redirect('/clientes/' . $client->id . '/editar')->with('status', 'cliente ' . $request->nickname . ' criado com sucesso.');
	}

	/**
	 * Atualiza o nickname do client removendo todas as acentuações
	 *
	 * @param Request $request
	 * @param $id
	 */
	public function editarNicknameClient(Request $request, $id)
	{
		if (!auth()->user()->isMaster()) {
			return response()->json([
				'success' => false,
				'message' => 'Você não tem permissão para acessar essa área.',
			], 403);
		}

		$validate = Validator::make($request->all(), [
			'nickname' => ['required', 'max:100'],
			'id' => ['required', 'exists:clients,id', 'in:' . $id],
			'taxnumber' => ['required', 'exists:clients,taxnumber'],
		], [
			'nickname.required' => 'O novo nome é obrigatório.',
			'nickname.max' => 'O novo nome deve ter no máximo 100 caracteres.',
			'multisafe_id.required' => 'O id do registro é obrigatório.',
			'multisafe_id.exists' => 'O id do registro não existe.',
			'multisafe_id.in' => 'O id do registro não é válido.',
			'client_id.required' => 'O id do cliente é obrigatório.',
			'client_id.exists' => 'O id do cliente não existe.',
			'maintaxnumber.required' => 'O CNPJ do cliente é obrigatório.',
			'maintaxnumber.exists' => 'O CNPJ do cliente não existe.',
		]);

		if ($validate->fails()) {
			return back()->with('warning', 'Erro: ' . $validate->errors()->first());
		}

		$client = Client::find($id);

		$newNickname = $request->nickname;
		$normalized = Normalizer::normalize($newNickname, Normalizer::NFD);
		$nickname = preg_replace('/[[`~´^]/', '', preg_replace('/[\x{0300}-\x{036F}]/u', '', $normalized));

		$client->nickname = $nickname;

		if (!$client->save()) {
			return back()->with('warning', 'Erro: não foi possível atualizar o nome do cliente.');
		}

		return back()->with('success', 'Nome do cliente atualizado com sucesso.');
	}

	/**
	 * Recebe os dados da view e edita o cliente.
	 * @param Request $request Dados da view
	 * @param string $id Id do cliente na tabela Clients
	 */
	public function editarCliente(Request $request, $id)
	{
		if (Auth::user()->role_id > 1)
			return redirect('/home')->with('warning', 'Você não tem permissão para acessar a área selecionada.');

		$client = Client::find($id);

		$spsInfo = getSPS();

		$users = User::select('id', 'name', 'email', 'role_id')->where('client_id', '=', $id)->get();

		$curlParams = "{ \"Method\": \"GetAccount\",
                    \"PartnerId\": " . $spsInfo->partnerid . ",
                    \"BusinessUnitId\": " . $spsInfo->businessunitid . ",
                    \"TaxNumber\": \"" . $client->taxnumber . "\"
                }";

		$clientfitbank = sendCurl($curlParams);
		$roles = Role::where('id', '>', 1)->get();

		$bancos = Bank::all();

		if ($clientfitbank["Success"] == "false") {
			$clientfitbank = false;
			return view('clients.edit', compact('client', 'users', 'clientfitbank', 'roles', 'bancos'))->with('warning', 'Erro: ocorreu um problema ao processar a requisição.');
		}

		return view('clients.edit', compact('client', 'users', 'clientfitbank', 'roles', 'bancos'));
	}

	public function update(Request $request, $id)
	{
		$client = Client::findOrFail($request->id);

		$request->merge([
			'payments' => covert2Float($request->payments),
			'transfers' => covert2Float($request->transfers),
			'billets' => covert2Float($request->billets),
			'balanceusage' => covert2Float($request->balanceusage),
			'monthlyfee' => covert2Float($request->monthlyfee)
		]);

		$client->update($request->all());

		if (!$client)
			return view('clients.vercliente', compact('client'))->with('warning', 'Houve um problema ao atualizar. Por favor tente mais tarde.');

		return view('clients.vercliente', compact('client'));
	}

	/**
	 * Atualiza o nickname do multisafe removendo todas as acentuações
	 *
	 * @param Request $request
	 * @param $id
	 * @param $multisafe
	 */
	public function editarNicknameMultisafe(Request $request, $id, $multisafe)
	{
		if (!auth()->user()->isMaster()) {
			return response()->json([
				'success' => false,
				'message' => 'Você não tem permissão para acessar essa área.',
			], 403);
		}

		$validate = Validator::make($request->all(), [
			'nickname' => ['required', 'max:100'],
			'multisafe_id' => ['required', 'exists:multisafes,id', 'in:' . $multisafe],
			'client_id' => ['required', 'exists:clients,id', 'in:' . $id],
			'maintaxnumber' => ['required', 'exists:clients,taxnumber'],
		], [
			'nickname.required' => 'O novo nome é obrigatório.',
			'nickname.max' => 'O novo nome deve ter no máximo 100 caracteres.',
			'multisafe_id.required' => 'O id do registro é obrigatório.',
			'multisafe_id.exists' => 'O id do registro não existe.',
			'multisafe_id.in' => 'O id do registro não é válido.',
			'client_id.required' => 'O id do cliente é obrigatório.',
			'client_id.exists' => 'O id do cliente não existe.',
			'maintaxnumber.required' => 'O CNPJ do cliente é obrigatório.',
			'maintaxnumber.exists' => 'O CNPJ do cliente não existe.',
		]);

		if ($validate->fails()) {
			return back()->with('warning', 'Erro: ' . $validate->errors()->first());
		}

		$multisafe = Multisafe::find($multisafe);

		$newNickname = $request->nickname;
		$normalized = Normalizer::normalize($newNickname, Normalizer::NFD);
		$nickname = preg_replace('/[[`~´^]/', '', preg_replace('/[\x{0300}-\x{036F}]/u', '', $normalized));

		$multisafe->nickname = $nickname;

		if (!$multisafe->save()) {
			return back()->with('warning', 'Erro: não foi possível atualizar o nome do cliente.');
		}

		return back()->with('success', 'Nome do cliente atualizado com sucesso.');
	}

	public function sincronizacliente(Request $request, $id)
	{

		if (Auth::user()->role_id > 1)
			return redirect('/home')->with('warning', 'Você não tem permissão para acessar a área selecionada.');

		$client = Client::findOrFail($id);
		$client->update($request->all());

		return redirect('/clientes/' . $id . '/editar');
	}


	public function enviaclientefitbank($id)
	{
		$client = Client::where('id', '=', $id)->first();

		$spsInfo = getSPS();

		$curlParams = "{\"Method\": \"CreateAccount\",
                            \"PartnerId\": " . $spsInfo->partnerid . ",
                            \"BusinessUnitId\": " . $spsInfo->businessunitid . ",
                            \"Mail\": \"" . $client->mail . "\",
                            \"Phone\": \"" . $client->phone . "\",
                            \"Nickname\": \"" . $client->nickname . "\",
                            \"Name\": \"" . $client->name . "\",
                            \"TaxNumber\": \"" . $client->taxnumber . "\"
                        }";

		$response = sendCurl($curlParams);

		if ($response["Success"] == "false")
			return redirect('/clientes')->with('warning', 'Erro: ' . $response['Message']);

		$atualizaIdentifier = $this->updateBusinessUnitId($id, $response["Identifier"], false);

		return redirect('/clientes')->with('status', 'Enviado com sucesso ao FitBank');

	}


	public function enviaclientemultifitbank($cnpj)
	{
		if (strlen($cnpj) > 12)
			$client = Multisafe::where('taxnumber', '=', $cnpj)->first();
		else
			$client = Multisafe::where('id', '=', $cnpj)->first();

		$spsInfo = getSPS();

		$curlParams = "{\"Method\": \"CreateAccount\",
                            \"PartnerId\": " . $spsInfo->partnerid . ",
                            \"BusinessUnitId\": " . $spsInfo->businessunitid . ",
                            \"Mail\": \"" . $client->mail . "\",
                            \"Phone\": \"" . $client->phone . "\",
                            \"Nickname\": \"" . $client->nickname . "\",
                            \"Name\": \"" . $client->nickname . "\",
                            \"TaxNumber\": \"" . $client->taxnumber . "\"
                        }";

		$response = sendCurl($curlParams);

		if ($response["Success"] == "false")
			return redirect('/clientes')->with('warning', 'Erro: ' . $response['Message']);

		$atualizaIdentifier = $this->updateBusinessUnitId($client->id, $response["Identifier"], true);

		return redirect('/clientes')->with('status', 'Enviado com sucesso ao FitBank');

	}


	public function updateBusinessUnitId($id, $businessunitid, $multi = false)
	{
		if (!$multi) {
			$client = Client::findOrFail($id);
			$client->businessunitid = $businessunitid;
			$client->update();
		} else {
			$client = Multisafe::findOrFail($id);
			$client->businessunitid = $businessunitid;
			$client->update();
		}

		return true;
	}

	public function transferenciainterna()
	{

		$spsInfo = getSPS();
		$curlParams = "{ \"Method\": \"GetBanks\",
                    \"PartnerId\": " . $spsInfo->partnerid . ",
                    \"BusinessUnitId\": " . $spsInfo->businessunitid . "
                }";

		$buscabancos = sendCurl($curlParams);

		$bancos = ($buscabancos["Success"] == "false") ? false : $buscabancos["Banks"];

		/* consultar saldo atual do cliente */
		$client = getClientById(Auth::user()->client_id);

		$client_taxnumber = Session::get('cnpj');

		$datade = date('Y/m/d', strtotime('-7 days'));
		$dataate = date('Y/m/d');
		$curlParams = "{ \"Method\": \"GetAccountEntry\",
                            \"PartnerId\": " . $spsInfo->partnerid . ",
                            \"BusinessUnitId\": " . $spsInfo->businessunitid . ",
                            \"TaxNumber\": \"" . $client_taxnumber . "\",
                            \"StartDate\": \"" . $datade . "\",
                            \"EndDate\": \"" . $dataate . "\"
                        }";

		$extrato = sendCurl($curlParams);
		$msg = false;
		if ($extrato["Success"] == "false") {
			$msg = 'Ocorreu um problema ao solicitar o saldo SPS.';
			$extrato['Balance'] = 0;
		}
		/* /consultar saldo atual do cliente */

		//return view('clients.transferenciabancaria',compact('bancos','extrato'))->with('status',$msg);

		// Cálculo das taxas para adiantamento, considerando o total do saldoCompensar
		$clientMultiCnpj = getMultiCnpj($client->id);
		$clientMultiCnpjAvailableLimit = getAvailableLimitMultiCnpj($client_taxnumber);
		$saldocompensar = (empty($clientMultiCnpj)) ? $client->availablelimit : $clientMultiCnpjAvailableLimit;
		if ($client->transferspercentage == 1) {
			$taxaTransferencia = ($client->transfers * ($extrato['Balance'] + $saldocompensar)) / 100;
		} else {
			$taxaTransferencia = $client->transfers;
		}
		if ($client->balanceusagepercentage == 1) {
			$taxaAdiantamento = ($client->balanceusage * $saldocompensar) / 100;
		} else {
			$taxaAdiantamento = $client->balanceusage;
		}
		$taxaTransferencia = 0;
		// Calcula o valor do limite da operação
		$limiteoperacao = $extrato['Balance'] + $saldocompensar - $taxaTransferencia - $taxaAdiantamento;
		if ($limiteoperacao < 0) {
			$limiteoperacao = 0;
		}

		// Lista os CNPJs vinculados ao cliente logado para usar como conta destino
		$client_id = Auth::user()->client_id;
		$clients = Client::select('id', 'taxnumber', 'name', 'availablelimit')
			->where('id', '=', $client_id)
			->orderBy('name')
			->get();
		foreach ($clients as $k => $cli) {
			$clients[$k]->multiconta = getMultiCnpj($cli->id);
		}

		$cnpj_emuso = Session::get('cnpj');
		return view('clients.transferenciainterna', compact('clients', 'bancos', 'extrato', 'saldocompensar', 'limiteoperacao', 'cnpj_emuso'))->with('status', $msg);
	}

	public function realizaTransferenciaInterna(Request $request)
	{

		$client = Client::select('clients.*')
			->where('id', '=', Auth::user()->client_id)
			->first();
		$taxnumber_origem = Session::get('cnpj');
		$taxnumber_destino = $request->taxnumber_destino_id;

		$valor = covert2Float($request->amount);
		$identificador = $request->identificador;
		$cashbackvalue = 0;
		$data = date('Y/m/d');
		// Recupera saldo do cliente no FIT
		$saldoCliente = saldoCliente($taxnumber_origem);

		// Calcula limite da operação de acordo com opção de adiantamento
		$clientMultiCnpj = getMultiCnpj($client->id);
		$clientMultiCnpjAvailableLimit = getAvailableLimitMultiCnpj($taxnumber_origem);
		$saldocompensar = (empty($clientMultiCnpj)) ? $client->availablelimit : $clientMultiCnpjAvailableLimit;
		if ($client->transferspercentage == 1) {
			$taxaTransferencia = ($client->transfers * ($saldoCliente + $saldocompensar)) / 100;
		} else {
			$taxaTransferencia = $client->transfers;
		}
		if ($client->balanceusagepercentage == 1) {
			$taxaAdiantamento = ($client->balanceusage * $saldocompensar) / 100;
		} else {
			$taxaAdiantamento = $client->balanceusage;
		}
		$taxaTransferencia = 0;
		// - verifica opção de adiantamento para calcular limite
		if ($request->limiteespecial == 1) {
			$limiteoperacao = $saldoCliente + $saldocompensar - $taxaTransferencia - $taxaAdiantamento;
		} else {
			$limiteoperacao = $saldoCliente;
		}
		if ($limiteoperacao < 0) {
			$limiteoperacao = 0;
		}

		if ($valor > $limiteoperacao) {
			return redirect('/transferenciainterna')->with('warning', 'O valor a ser transferido é maior que o saldo disponível na conta de origem.');
		}

		//------------------------------
		$client_taxnumber = $taxnumber_origem;

		// CONSULTA O SALDO DO CLIENTE
		$saldoDisponivel = getAvailableLimitMultiCnpj($taxnumber_origem);
		$logLimitUsed = false;

		$saldoCompensar = 0;
		$saldoCompensarValor = 0;
		$internaltransferdocumentnumber = 0;
		$cashbackvalue = 0;

		$cli_ratevalue = consultaTaxaCliente($client->id, 'moneytransfer', covert2Float($valor));
		$cli_ratevalue = 0;
		// ratevaluetype:  0 - Enviado (Taxa enviada na requisição); 1 - Padrão (Taxa configurada na Unidade de negócio);2 - Nenhum (Sem Taxa)

		// verifica se o disponível é >= valor a ser transferido
		if ($valor > $saldoCliente) {
			// se não for, verifica se permite usar saldo a compensar
			if (isset($request->limiteespecial) && $request->limiteespecial == 1) {

				// saldo a transferir para completar o pagamento
				$saldoATransferir = ($valor - $saldoCliente);

				$clientSpsService = new ClientSpsService($taxnumber_origem);
				$cashbackvalue = $clientSpsService->calculaTaxaAdiantamentoParaOperacao('TRANSFERENCIA_INTERNA', $saldoATransferir);

				// verifica se o cliente possui saldo a compensar suficiente
				if (($saldoCliente + $saldoDisponivel) < ($valor + $cashbackvalue + $cli_ratevalue)) {
					return redirect('transferenciainterna')->with('warning', 'Não há saldo suficiente para realizar a operação.');
				} else {
					// realiza a transferencia do valor adicional para o cliente
					$transfereSaldoDisponivel = transfereSaldoDisponivelCliente($taxnumber_origem, $saldoATransferir, 'Transferencia de saldo a compensar', $cashbackvalue, $cli_ratevalue);
					if (!$transfereSaldoDisponivel)
						return redirect('transferenciainterna')->with('warning', 'Ocorreu um erro ao realizar a operação. Por favor tente mais tarde.');
					// - incluit o adiantamento no saldo devedor (usedavailabellimits)
					$transactionNumber = 'temp_' . $transfereSaldoDisponivel;
					$internaltransferdocumentnumber = $transfereSaldoDisponivel;
					$saldoDisponivel = 0; //getAvailableLimitMultiCnpj($client_taxnumber);
					$limitusedID = logAvailableLimitUsed(Auth::user()->id, $client->id, $taxnumber_origem, $saldoDisponivel, $saldoATransferir, 'TRANSFERENCIA INTERNA', $transactionNumber, false, $internaltransferdocumentnumber, $cashbackvalue);

					$logLimitUsed = true;
				}
			} else {
				return redirect('transferenciainterna')->with('warning', 'Não há saldo suficiente para realizar a operação.');
			}
		}
		// /CONSULTA O SALDO DO CLIENTE
		$cashbackValueEntreContas = 0;
		$transfere = transfereSaldoEntreContas($taxnumber_origem, $taxnumber_destino, $valor, 'Transferencia entre contas', $cashbackValueEntreContas);
		if (!$transfere)
			return redirect('/transferenciainterna')->with('warning', 'Erro: ocorreu um problema ao processar a requisição. Por favor tente mais tarde.');

		// atualiza o numero da transação FIT no limit utilizado
		if ($logLimitUsed) {
			$transactionNumber = (isset($transfere['DocumentNumber'])) ? $transfere['DocumentNumber'] : '0';
			$limiteusado = Usedavailablelimit::where('id', '=', $limitusedID)->first();
			$limiteusado->documentnumber = $transactionNumber;
			$limiteusado->update();
		}

		//return redirect('/transferenciainterna')->with('success', 'ORIGEM = '.$taxnumber_origem.', DESTINO = '.$taxnumber_destino.', valor = '.$valor.', identificador = '.$identificador.', Limite = '.$limiteoperacao);

		$data = date('d/m/Y H:i:s');
		return view('clients.transferenciainternaok', compact('transfere', 'taxnumber_origem', 'taxnumber_destino', 'valor', 'identificador', 'data'))->with('status', 'Transferência efetuada.');
	}

	/* ********************** ADMIN ********************** */
	public function transferenciacliente()
	{
		if (Auth::user()->role_id > 1)
			return redirect('/home')->with('warning', 'Você não tem permissão para acessar essa área.');

		$clients = Client::select('id', 'taxnumber', 'name', 'availablelimit')->orderBy('name')->get();

		$spsInfo = getSPS();

		$spsInfo = getSPS();
		$datade = date('Y/m/d', strtotime('-7 days'));
		$dataate = date('Y/m/d');
		$curlParams = "{ \"Method\": \"GetAccountEntry\",
                            \"PartnerId\": " . $spsInfo->partnerid . ",
                            \"BusinessUnitId\": " . $spsInfo->businessunitid . ",
                            \"TaxNumber\": \"" . $spsInfo->cnpj . "\",
                            \"StartDate\": \"" . $datade . "\",
                            \"EndDate\": \"" . $dataate . "\"
                        }";

		$extrato = sendCurl($curlParams);

		$msg = false;

		if ($extrato["Success"] == "false") {
			$msg = 'Ocorreu um problema ao solicitar o saldo SSG.';
			$extrato['Balance'] = 0;
		}


		foreach ($clients as $k => $cli) {
			$clients[$k]->multiconta = getMultiCnpj($cli->id);
		}


		return view('clients.transferenciacliente', compact('clients', 'extrato'))->with('status', $msg);
	}

	public function transferenciaSimplesCliente() // carrega a pagina que fará a transferencia
	{
		if (!auth()->user()->isMaster()) {
			return redirect('/home')->with('warning', 'Você não tem permissão para acessar essa área.');
		}

		$spsInfo = getSPS();
		$oneWeekAgo = date('Y/m/d', strtotime('-7 days'));
		$today = date('Y/m/d');
		$curlParams = [
			"Method" => "GetAccountEntry",
			"PartnerId" => $spsInfo->partnerid,
			"BusinessUnitId" => $spsInfo->businessunitid,
			"TaxNumber" => $spsInfo->cnpj,
			"StartDate" => $oneWeekAgo,
			"EndDate" => $today,
		];

		$extrato = sendCurl(json_encode($curlParams));

		$msg = false;
		if ($extrato["Success"] == "false") {
			$msg = 'Ocorreu um problema ao solicitar o saldo SSG.';
			$extrato['Balance'] = 0;
		}

		return view('clients.transferencia.cliente', compact('extrato'))
			->with('status', $msg);
	}

	public function getClientDetails(Request $request): string
	{
		if (!auth()->user()->isMaster()) {
			return redirect('/home')->with('warning', 'Você não tem permissão para acessar essa área.');
		}

		$clients = Client::query()
			->select('id', 'name', 'taxnumber', 'availablelimit')
			->when($request->has('name'), function (Builder $query) use ($request) {
				$query->where('name', 'like', "%{$request->name}%");
			})
			->where('active', 1)
			->orderBy('name')
			->get();

		$multisafes = Multisafe::query()
			->select('id', 'nickname', 'taxnumber', 'availablelimit')
			->when($request->has('name'), function (Builder $query) use ($request) {
				$query->where('nickname', 'like', "%{$request->name}%");
			})
			->where('active', 1)
			->orderBy('nickname')
			->get();

		$merge = array_merge($clients->toArray(), $multisafes->toArray());

		$response = array_map(function ($client) {
			return [
				'id' => $client['id'],
				'name' => $client['name'] ?? $client['nickname'],
				'taxnumber' => $client['taxnumber'],
				'availablelimit' => $client['availablelimit'],
			];
		}, $merge);

		return json_encode([
			'success' => true,
			'data' => $response,
		]);
	}


	public function realizaTransferenciaCliente(Request $request) //FAZ A TRANSFERENCIA
	{
		if (!auth()->user()->isMaster()) {
			return redirect('/home')->with('warning', 'Você não tem permissão para acessar essa área.');
		}

		$request->validate([
			'taxnumber' => 'required',
			'valor' => 'required',
		], [
			'taxnumber.required' => 'É necessário informar um cliente',
			'valor.required' => 'É necessário informar o valor da transferência',
		]);

		// $client = Client::where('id','=',$request->client_id)->first();

		$client_availablelimit = getAvailableLimitMultiCnpj($request->taxnumber);


		$transfRapida = strpos($request->headers->get('referer'), 'transferenciarapida');

		if (covert2Float($request->valor) > $client_availablelimit) {
			if ($transfRapida !== false)
				return redirect('/transferenciarapida')->with('warning', 'O valor a ser transferido é maior que o saldo disponível do cliente.');

			return redirect('/transferencia-cliente')->with('warning', 'O valor a ser transferido é maior que o saldo disponível do cliente.');
		}

		$transfere = transfereSaldoDisponivelCliente(somenteNumeros($request->taxnumber), covert2Float($request->valor), 'Transferencia interna', 0, 0, true);
		Session::put('transfere',$transfere);
		if (!$transfere)
			return redirect('/transferencia-cliente')->with('warning', 'Erro: ocorreu um problema ao processar a requisição. Por favor tente mais tarde.');

		if ($transfRapida !== false)
			return redirect('/transferenciarapida')->with('success', 'Transferência efetuada com sucesso.');

		// return view('clients.transferenciaclienteok',compact('transfere','client'))->with('status', 'Transferência efetuada.');
		// return view('clients.transferenciaclienteok', compact('transfere'))->with('status', 'Transferência efetuada.');
		return redirect('telatransferenciarealizada');

	}

	public function transferenciaClienteRealizada(Request $request){
		$transfere = Session::get('transfere');
		return view('clients.transferenciaclienteok', compact('transfere'))->with('status', 'Transferência efetuada.');
	}

	public function realizaTransferenciaClienteRapida(Request $request)
	{
		// Recupera valores informados no formulário
		$taxnumber = $request->taxnumber;
		$realcollectedvalue = covert2Float($request->inputColeta);
		$transferedvalue = covert2Float($request->valor);

		// Realizar a transferencia (SPS -> Taxnumber)
		if ($transferedvalue > 0) {
			//$transactionNumber = 1; //transfereSaldoDisponivelCliente
			$transactionNumber = transfereSaldoDisponivelCliente(somenteNumeros($taxnumber), $transferedvalue, 'Transferencia interna');

			// Se transferência falhar: finalizar processamento e exibir mensagem de alerta
			if (!$transactionNumber)
				return redirect('/transferenciarapida')->with('status', 'Ocorreu um erro ao realizar a operação. Por favor tente mais tarde.');
		} else {
			$transactionNumber = 'SEM DISTRIBUICAO';
		}

		// Se transferência OK: incluir coleta
		// - incluir coleta
		$collectID = saveRealCollect($taxnumber, $realcollectedvalue, $transferedvalue, $transactionNumber);
		// - se falhar na criação da coleta real: finalizar processamento com erro
		if (!$collectID)
			return redirect('/transferenciarapida')->with('warning', 'Favor entrar em contato com ENGENHARIA. Transferência realizada com FALHA na atualização do Saldo Devedor.');
		// - se criação da coleta real OK : pagar saldo devedor e atualizar collectsmap
		//   - pagar saldo devedor
		$valorParaPagamento = $realcollectedvalue - $transferedvalue;
		$paymentvalue = pagarSaldoDevedor_Zane($collectID, $taxnumber, $valorParaPagamento);
		$differencevalue = $valorParaPagamento - $paymentvalue;
		if ($differencevalue < 0.01) {
			$closed = 1;
			$differencevalue = 0;
		} else {
			$closed = 0;
		}
		//   - atualizar collectsmap
		$coletaReal = CollectsMap::where('id', '=', $collectID)->first();
		$coletaReal->paymentvalue = $paymentvalue;
		$coletaReal->differencevalue = $differencevalue;
		$coletaReal->closed = $closed;
		$coletaReal->update();
		//   - finalizar processamento de transferência com sucesso
		return redirect('/transferenciarapida')->with('success', 'TESTANDO CASO DE SUCESSO => ' . $collectID);
		//return redirect('/transferenciarapida')->with('warning', 'AINDA NÃO LIBERADO PARA HOMOLOGAÇÃO.');
	}

	public function viewAtivarD1Automatico(int $idCliente, int $idMultisafe = 0)
	{
		return $this->viewAtivarDAutomatico($idCliente, $idMultisafe, "D+1");
	}

	public function viewAtivarD0Automatico(int $idCliente, int $idMultisafe = 0)
	{
		return $this->viewAtivarDAutomatico($idCliente, $idMultisafe, NULL);
	}

	/**
	 * @param int $idCliente Id do cliente na tabela Clients
	 * @param int $idMultisafe Id do registro que está sendo atualizado, podendo ser da tabela
	 * Clients ou Multisafes
	 * @param ?string $ehD1 Se a operação for do tipo D+1, esta variável existirá e terá o valor "d1"
	 * @return \Illuminate\Contracts\Foundation\Application|\Illuminate\Contracts\View\Factory|\Illuminate\View\View
	 */
	public function viewAtivarDAutomatico(int $idCliente, int $idMultisafe, ?string $ehD1)
	{
		// 1) Checar se existe $idMultisafe ou não
		$ehMultisafe = FALSE;
		if ($idMultisafe != 0) {
			$objCliente = Multisafe::find($idMultisafe);
			$ehMultisafe = TRUE;
		} else {
			$objCliente = Client::find($idCliente);
		}

		if (!$objCliente) {
			return back()->with('warning', 'Dados inconsistentes. Por favor faça login novamente.');
		}

		// 2) Obter os dados da tabela banks
		$bancos = Bank::all();

		return view('clients.d0automatico', [
			'cliente' => $objCliente,
			'bancos' => $bancos,
			'ehMultisafe' => $ehMultisafe,
			'ehD1' => $ehD1
		]);
	}

	/**
	 * Identifica o cliente e ativa a funcionalidade D+0 automático
	 * @param int $idCliente Id do cliente na tabela Clients
	 * @param int $idMultisafe Id do registro que está sendo ativado, podendo ser da tabela
	 * Clients ou Multisafes
	 */
	public function ativarDAutomatico(int $idCliente, Request $request)
	{
		$objCliente = buscarClienteOuMultisafePeloTaxnumber($request["fromtaxnumber"]);

		$dadosDestinatario = [
			"fromtaxnumber" => $request["fromtaxnumber"],
			"totaxnumber" => $request["totaxnumber"],
			"identifier" => $request["toidentifier"],
			"accounttype" => $request["accounttype"],
			"nickname" => $request["tonickname"],
			"bank_id" => (int) $request["bank_id"],
			"bankbranch" => $request["bankbranch"],
			"bankaccount" => $request["bankaccount"],
			"bankaccountdigit" => $request["bankaccountdigit"],
			"operationtype" => $request["operationtype"]
		];

		try {
			$novoDestinatario = new Autotransferreceiver();
			$novoDestinatario->fill($dadosDestinatario);
			$novoDestinatario->save();

			if ($request["operationtype"] == 'D+0') {
				$objCliente->d0automatic = "1";
				$objCliente->d0time = $request["operationtime"] . ":00";
			} elseif ($request["operationtype"] == 'D+1') {
				$objCliente->d1automatic = "1";
				$objCliente->d1time = $request["operationtime"] . ":00";
			}
			$objCliente->save();

			return redirect("/clientes/${idCliente}/editar")->with('success', 'Operação realizada.');
		} catch (\Exception $e) {
			customLog("Ativação DX Automático: " . $e->getMessage());
			customLog("Ativação DX Automático: " . json_encode($request->all()));
			return back()->with('warning', 'Operação não realizada. Por favor, revise os dados inseridos.');
		}

	}

	public function buscarDocumento(Request $request)
	{
		$objCliente = buscarClienteOuMultisafePeloTaxnumber($request['taxnumber']);
		if ($objCliente) {
			return response()->json([
				'nickname' => $objCliente->nickname
			]);
		}
	}

	public function desativarD0Automatico(int $idCliente, int $idMultisafe = 0)
	{
		return $this->desativarDAutomatico($idCliente, $idMultisafe, "D+0");
	}

	public function desativarD1Automatico(int $idCliente, int $idMultisafe = 0)
	{
		return $this->desativarDAutomatico($idCliente, $idMultisafe, "D+1");
	}

	/**
	 * Identifica o cliente e desativa a funcionalidade D+0 automático
	 * @param int $idCliente Id do cliente na tabela Clients
	 * @param int $idMultisafe Id do registro que está sendo desativado, podendo ser da tabela
	 * Clients ou Multisafes
	 */
	public function desativarDAutomatico(int $idCliente, int $idMultisafe, string $operationtype)
	{
		if ($idMultisafe == 0) {
			$objCliente = Client::find($idCliente);
		} else {
			$objCliente = Multisafe::find($idMultisafe);
		}

		$objDestinatario = Autotransferreceiver::where("fromtaxnumber", "=", $objCliente->taxnumber)
			->where("operationtype", "=", $operationtype)
			->first();

		try {

			if ($operationtype == "D+0") {
				$objCliente->d0automatic = "0";
				$objCliente->d0time = "00:00:00";
			} elseif ($operationtype == "D+1") {
				$objCliente->d1automatic = "0";
				$objCliente->d1time = "00:00:00";
			}
			$objCliente->save();

			if ($objDestinatario)
				$objDestinatario->delete();

			return back()->with('success', 'Operação realizada.');
		} catch (Exception $e) {
			return back()->with('warning', 'Operação não realizada. Por favor, tente novamente.');
		}
	}

	/* ********************** /ADMIN ********************** */

	public function mudaStatusCliente(Request $request, $id)
	{
		$client = Client::findOrFail($id);

		try {

			if ($client->d0automatic == "1") {
				$this->desativarD0Automatico($id, 0);
			}

		} catch (\Exception $e) {
			// TODO
			// Checar com Robson caso não consiga desativar as operações
		}

		try {

			if ($client->d1automatic == "1") {
				$this->desativarD1Automatico($id, 0);
			}

		} catch (\Exception $e) {
			// TODO
			// Checar com Robson caso não consiga desativar as operações
		}

		$client->active = ($client->active == 1) ? 0 : 1;
		$client->update();

		$mensagem = ($client->active == 1) ? 'O cliente foi habilitado.' : 'O cliente foi desativado.';

		$clients = Client::orderBy('name')
			->orderBy('active', 'desc')
			->orderBy('name', 'asc')
			->get();
		return view('clients.index', compact('clients'))->with('status', $mensagem);
	}

	/* ********************** TRANSFERENCIA BANCÁRIA ********************** */

	public function transferenciabancariaincluir()
	{

		$spsInfo = getSPS();

		$curlParams = "{ \"Method\": \"GetBanks\",
                    \"PartnerId\": " . $spsInfo->partnerid . ",
                    \"BusinessUnitId\": " . $spsInfo->businessunitid . "
                }";

		$buscabancos = sendCurl($curlParams);

		$bancos = ($buscabancos["Success"] == "false") ? false : $buscabancos["Banks"];

		// consultar saldo atual do cliente
		$client = getClientById(Auth::user()->client_id);

		$client_taxnumber = Session::get('cnpj');

		$datade = date('Y/m/d', strtotime('-7 days'));
		$dataate = date('Y/m/d');
		$curlParams = "{ \"Method\": \"GetAccountEntry\",
                            \"PartnerId\": " . $spsInfo->partnerid . ",
                            \"BusinessUnitId\": " . $spsInfo->businessunitid . ",
                            \"TaxNumber\": \"" . $client_taxnumber . "\",
                            \"StartDate\": \"" . $datade . "\",
                            \"EndDate\": \"" . $dataate . "\"
                        }";

		$extrato = sendCurl($curlParams);
		$msg = false;
		if ($extrato["Success"] == "false") {
			$msg = 'Ocorreu um problema ao solicitar o saldo SPS.';
			$extrato['Balance'] = 0;
		}
		//consultar saldo atual do cliente

		//return view('clients.transferenciabancaria',compact('bancos','extrato'))->with('status',$msg);

		//ZANE (1.1.0)
		$clientMultiCnpj = getMultiCnpj($client->id);
		$clientMultiCnpjAvailableLimit = getAvailableLimitMultiCnpj($client_taxnumber);
		$saldocompensar = (empty($clientMultiCnpj)) ? $client->availablelimit : $clientMultiCnpjAvailableLimit;
		if ($client->transferspercentage == 1) {
			$taxaTransferencia = ($client->transfers * ($extrato['Balance'] + $saldocompensar)) / 100;
		} else {
			$taxaTransferencia = $client->transfers;
		}
		if ($client->balanceusagepercentage == 1) {
			$taxaAdiantamento = ($client->balanceusage * $saldocompensar) / 100;
		} else {
			$taxaAdiantamento = $client->balanceusage;
		}
		//$taxaAdiantamento = $saldocompensar *
		$limiteoperacao = $extrato['Balance'] + $saldocompensar - $taxaTransferencia - $taxaAdiantamento;

		return view('clients.transferenciabancaria.inclusaoinfo', compact('bancos', 'extrato', 'saldocompensar', 'limiteoperacao'))->with('status', $msg);

	}

	public function transferenciabancariarealizarinclusao(Request $request)
	{

		$client = getClientById(Auth::user()->client_id);

		$spsInfo = getSPS();

		$client_taxnumber = Session::get('cnpj');

		// CONSULTA O SALDO DO CLIENTE
		$saldoCliente = saldoCliente($client_taxnumber);
		$saldoDisponivel = getAvailableLimitMultiCnpj($client_taxnumber);
		// valor a ser transferido
		$valor = covert2Float($request->valor);
		$logLimitUsed = false;

		$newIdentifier = concatIdentifier($request->identificador);

		$curlParams = '';
		$cli_ratevalue = consultaTaxaCliente($client->id, 'moneytransfer', covert2Float($valor));
		$transactionNumber = '0';
		$url = '';
		$alreadyexists = 0;
		$saldoCompensar = 0;
		$saldoCompensarValor = 0;
		$internaltransferdocumentnumber = 0;
		$cashbackvalue = 0;
		//$entryid = '';
		//$validacao = '';

		$mensagem = '';

		$limiteespecial = (isset($request->limiteespecial) && $request->limiteespecial == 1) ? '1' : '0';
		$valorlimiteespecial = 0;

		$sucesso = 1;
		$resultado = saveTransferManager(
			$client_taxnumber,
			somenteNumeros($request->documento),
			$request->nome,
			somenteNumeros($request->banco),
			somenteNumeros($request->agencia),
			somenteNumeros($request->conta),
			somenteNumeros($request->digito),
			somenteNumeros($request->tipoconta),
			$valor,
			$cli_ratevalue,
			$newIdentifier,
			$sucesso,
			$transactionNumber,
			$url,
			$alreadyexists,
			$saldoCompensar,
			$saldoCompensarValor,
			$curlParams
		);

		$transfere['id'] = $resultado['id'];

		return view('clients.transferenciabancaria.inclusaook', compact('transfere'));
	}

	public function transferenciabancariaautorizarconsulta(Request $request)
	{
		return redirect()->action([ClientWebController::class, 'consultatransferenciabancaria'], ['filtrostatus' => 'Incluido']);
	}

	public function transferenciabancariaautorizar(Request $request)
	{
		$spsInfo = getSPS();

		$curlParams = "{ \"Method\": \"GetBanks\",
                    \"PartnerId\": " . $spsInfo->partnerid . ",
                    \"BusinessUnitId\": " . $spsInfo->businessunitid . "
                }";

		$buscabancos = sendCurl($curlParams);

		$bancos = ($buscabancos["Success"] == "false") ? false : $buscabancos["Banks"];

		/* consultar saldo atual do cliente */
		$client = getClientById(Auth::user()->client_id);

		$client_taxnumber = Session::get('cnpj');

		$datade = date('Y/m/d', strtotime('-7 days'));
		$dataate = date('Y/m/d');
		$curlParams = "{ \"Method\": \"GetAccountEntry\",
                            \"PartnerId\": " . $spsInfo->partnerid . ",
                            \"BusinessUnitId\": " . $spsInfo->businessunitid . ",
                            \"TaxNumber\": \"" . $client_taxnumber . "\",
                            \"StartDate\": \"" . $datade . "\",
                            \"EndDate\": \"" . $dataate . "\"
                        }";

		$extrato = sendCurl($curlParams);
		$msg = false;
		if ($extrato["Success"] == "false") {
			$msg = 'Ocorreu um problema ao solicitar o saldo SPS.';
			$extrato['Balance'] = 0;
		}

		//ZANE (1.1.0)
		$clientMultiCnpj = getMultiCnpj($client->id);
		$clientMultiCnpjAvailableLimit = getAvailableLimitMultiCnpj($client_taxnumber);
		$saldocompensar = (empty($clientMultiCnpj)) ? $client->availablelimit : $clientMultiCnpjAvailableLimit;
		if ($client->transferspercentage == 1) {
			$taxaTransferencia = ($client->transfers * ($extrato['Balance'] + $saldocompensar)) / 100;
		} else {
			$taxaTransferencia = $client->transfers;
		}
		if ($client->balanceusagepercentage == 1) {
			$taxaAdiantamento = ($client->balanceusage * $saldocompensar) / 100;
		} else {
			$taxaAdiantamento = $client->balanceusage;
		}
		//$taxaAdiantamento = $saldocompensar *
		$limiteoperacao = $extrato['Balance'] + $saldocompensar - $taxaTransferencia - $taxaAdiantamento;

		// RECUPERANDO INCLUSÃO DA TRANSFERENCIA
		$transfermanagerid = 1;
		if (isset($request->transfermanagerid)) {
			$transfermanagerid = $request->transfermanagerid;
		}
		$transferinfo = Transfermanager::where('id', '=', $transfermanagerid)->first();

		return view('clients.transferenciabancaria.autorizarinfo', compact('bancos', 'extrato', 'saldocompensar', 'limiteoperacao', 'transferinfo'))->with('status', $msg);
	}

	public function transferenciabancaria()
	{

		$spsInfo = getSPS();

		$curlParams = "{ \"Method\": \"GetBanks\",
                    \"PartnerId\": " . $spsInfo->partnerid . ",
                    \"BusinessUnitId\": " . $spsInfo->businessunitid . "
                }";

		$buscabancos = sendCurl($curlParams);

		$bancos = ($buscabancos["Success"] == "false") ? false : $buscabancos["Banks"];

		/* consultar saldo atual do cliente */
		$client = getClientById(Auth::user()->client_id);

		$client_taxnumber = Session::get('cnpj');

		$datade = date('Y/m/d', strtotime('-7 days'));
		$dataate = date('Y/m/d');
		$curlParams = "{ \"Method\": \"GetAccountEntry\",
                            \"PartnerId\": " . $spsInfo->partnerid . ",
                            \"BusinessUnitId\": " . $spsInfo->businessunitid . ",
                            \"TaxNumber\": \"" . $client_taxnumber . "\",
                            \"StartDate\": \"" . $datade . "\",
                            \"EndDate\": \"" . $dataate . "\"
                        }";

		$extrato = sendCurl($curlParams);
		$msg = false;
		if ($extrato["Success"] == "false") {
			$msg = 'Ocorreu um problema ao solicitar o saldo SPS.';
			$extrato['Balance'] = 0;
		}
		/* /consultar saldo atual do cliente */

		//return view('clients.transferenciabancaria',compact('bancos','extrato'))->with('status',$msg);

		$clientSPS = new ClientSpsService($client_taxnumber);

		$saldocompensar = $clientSPS->getSaldoCompensar();

		//ZANE (1.1.0)

		//$clientMultiCnpj = getMultiCnpj($client->id);
		//$clientMultiCnpjAvailableLimit = getAvailableLimitMultiCnpj($client_taxnumber);
		//$saldocompensar = (empty($clientMultiCnpj))? $client->availablelimit : $clientMultiCnpjAvailableLimit;

		//Limite da operação
		//Desconta taxas da transação e valor total provisionado do cliente

		$limiteoperacao = $clientSPS->calculaLimiteDaOperacao('TRANSFERENCIA', True);


		return view('clients.transferenciabancaria', compact('bancos', 'extrato', 'saldocompensar', 'limiteoperacao'))->with('status', $msg);

	}

	public function transferenciabancariacontato(Request $request)
	{

		$spsInfo = getSPS();

		$curlParams = "{ \"Method\": \"GetBanks\",
                    \"PartnerId\": " . $spsInfo->partnerid . ",
                    \"BusinessUnitId\": " . $spsInfo->businessunitid . "
                }";

		$buscabancos = sendCurl($curlParams);

		$bancos = ($buscabancos["Success"] == "false") ? false : $buscabancos["Banks"];

		/* consultar saldo atual do cliente */
		$client = getClientById(Auth::user()->client_id);

		$contact = Clientcontact::where('id', '=', $request->idcontato)->first();

		$client_taxnumber = Session::get('cnpj');

		$datade = date('Y/m/d', strtotime('-7 days'));
		$dataate = date('Y/m/d');
		$curlParams = "{ \"Method\": \"GetAccountEntry\",
                            \"PartnerId\": " . $spsInfo->partnerid . ",
                            \"BusinessUnitId\": " . $spsInfo->businessunitid . ",
                            \"TaxNumber\": \"" . $client_taxnumber . "\",
                            \"StartDate\": \"" . $datade . "\",
                            \"EndDate\": \"" . $dataate . "\"
                        }";

		$extrato = sendCurl($curlParams);
		$msg = false;
		if ($extrato["Success"] == "false") {
			$msg = 'Ocorreu um problema ao solicitar o saldo SPS.';
			$extrato['Balance'] = 0;
		}
		/* /consultar saldo atual do cliente */

		return view('clients.transferenciabancariacontato', compact('bancos', 'extrato', 'contact'))->with('status', $msg);

	}

	/*
																				  Data: 13/07/2022
																				  Autor: Fabio
																				  A função realizatransferenciabancaria() foi analisada em busca de possíveis erros que causassem a multiplicidade das operações de transferência.
																				  Não foram encontrados erros que justificassem essa multiplicidade das operações.
																				  Esta função realiza diversas operações, podendo demorar alguns segundos para performar a efetiva transferencia de recursos.
																				  A principal causa encontrada da multiplicidade é o clientes clicar diversas vezes no botão de "Realizar Operação"

																				  OBS.: Este mesmo problema pode está ocorrendo em outras operações realizadas no plataforma.
																				  */
	public function realizatransferenciabancaria(Request $request)
	{

		$client = getClientById(Auth::user()->client_id);

		$spsInfo = getSPS();


		if (!operationTimeLimit('transferencia') && isToday($request->datatransf)) {
			return redirect('transferencias')->with('warning', 'Transferências após o horário limite devem ser agendadas para o próximo dia útil.');
		}

		if (isWeekend($request->datatransf)) {
			return redirect('transferencias')->with('warning', 'Transferências nos finais de semana devem ser agendadas para o próximo dia útil.');
		}

		$client_taxnumber = Session::get('cnpj');

		// CONSULTA O SALDO DO CLIENTE
		$saldoCliente = saldoCliente($client_taxnumber);
		$saldoDisponivel = getAvailableLimitMultiCnpj($client_taxnumber);



		// valor a ser transferido
		$valor = covert2Float($request->valor);
		$logLimitUsed = false;
		$idAdiantamentoUtilizado = false;

		$saldoCompensar = 0;
		$saldoCompensarValor = 0;
		$internaltransferdocumentnumber = 0;
		$cashbackvalue = 0;

		$cli_ratevalue = consultaTaxaCliente($client->id, 'moneytransfer', covert2Float($valor));
		// ratevaluetype:  0 - Enviado (Taxa enviada na requisição); 1 - Padrão (Taxa configurada na Unidade de negócio);2 - Nenhum (Sem Taxa)

		// verifica se o disponível é >= valor a ser transferido
		if ($valor > $saldoCliente) {
			// se não for, verifica se permite usar saldo a compensar
			if (isset($request->limiteespecial) && $request->limiteespecial == 1) {

				// saldo a transferir para completar o pagamento
				$saldoATransferir = ($valor - $saldoCliente);

				// ao saldo a transferir deve ser adicionada a taxa
				$cashbackvalue = consultaTaxaCliente($client->id, 'internaltransfer', $saldoATransferir);

				// verifica se o cliente possui saldo a compensar suficiente
				if (($saldoCliente + $saldoDisponivel) < ($valor + $cashbackvalue + $cli_ratevalue)) {
					return redirect('transferencias')->with('warning', 'Não há saldo suficiente para realizar a operação.');

				} else {
					// realiza a transferencia do valor adicional para o cliente
					$transfereSaldoDisponivel = transfereSaldoDisponivelCliente($client_taxnumber, $saldoATransferir, 'Transferencia de saldo a compensar', $cashbackvalue, $cli_ratevalue);

					if (!$transfereSaldoDisponivel)
						return redirect('transferencias')->with('warning', 'Ocorreu um erro ao realizar a operação. Por favor tente mais tarde.');

					// $logLimitUsed = true;
					$idAdiantamentoUtilizado = logAvailableLimitUsed(Auth::user()->id, $client->id, $client_taxnumber, $saldoDisponivel, $saldoATransferir, 'ADIANTAMENTO_TRANSFERENCIA_EXTERNA', 0, false, $internaltransferdocumentnumber, $cashbackvalue);

					$saldoCompensar = 1;
					$saldoCompensarValor = $saldoATransferir;
					$internaltransferdocumentnumber = $transfereSaldoDisponivel;
				}
			} else {
				return redirect('transferencias')->with('warning', 'Não há saldo suficiente para realizar a operação.');
			}
		}


		/*
																																								    Data: 01/10/2022
																																								    Autor: Fabio

																																								    Correção da transferencia duplicada.
																																								    Foi criado um novo formato de identificador que é utilizado para verificar se a transferencia já foi solicitada
																																								    */

		//$newIdentifier = concatIdentifier($request->identificador);

		//Novo identificador da transação
		//Exemplo: 20221001__2342344
		$newIdentifier = date('Ymd', strtotime("now")) . '__' . $request->identificador;

		//busca no banco se há uma transferencia com o identificador informado
		$transferinfo = Transferlog::
			where('user_id', '=', Auth::user()->id)
			->where('taxnumber', '=', $client_taxnumber)
			->where('identificador', '=', $newIdentifier)
			->first();

		//Se houver uma transferencia com o identificador, redireciona o usuário informando a situação
		if ($transferinfo) {
			return redirect('transferencias')->with('warning', 'Já existe uma transferência com o identificador: ' . $request->identificador);
		}

		// $cli_ratevalue = consultaTaxaCliente($client->id, 'moneytransfer', covert2Float($valor));
		// // ratevaluetype:  0 - Enviado (Taxa enviada na requisição); 1 - Padrão (Taxa configurada na Unidade de negócio);2 - Nenhum (Sem Taxa)


		$curlParams = "{ \"Method\": \"MoneyTransfer\",
                    \"PartnerId\": " . $spsInfo->partnerid . ",
                    \"BusinessUnitId\": " . $spsInfo->businessunitid . ",
                    \"FromTaxNumber\": \"" . $client_taxnumber . "\",
                    \"ToTaxNumber\": \"" . somenteNumeros($request->documento) . "\",
                    \"ToName\": \"" . $request->nome . "\",
                    \"Bank\": \"" . somenteNumeros($request->banco) . "\",
                    \"BankBranch\": \"" . somenteNumeros($request->agencia) . "\",
                    \"BankAccount\": \"" . somenteNumeros($request->conta) . "\",
                    \"BankAccountDigit\": \"" . somenteNumeros($request->digito) . "\",
                    \"AccountType\": \"" . somenteNumeros($request->tipoconta) . "\",
                    \"Value\": " . $valor . ",
                    \"RateValue\": \"" . $cli_ratevalue . "\",
                    \"Identifier\": \"" . $newIdentifier . "\",
                    \"RateValueType\": \"0\",
                    \"PaymentDate\": \"" . dataSlashReverse($request->datatransf) . "\",
                    \"Tags\": []
                }";

		customLog('realizatransferenciabancaria: ' . $curlParams);

		$transfere = sendCurl($curlParams);
		
		if ($transfere["Success"] == "false") {
			$validacao = (isset($transfere['Validation'])) ? serialize($transfere['Validation']) : '';
			$mensagem = (isset($transfere['Message'])) ? $transfere['Message'] : '';

			saveTransferErrorLog('MoneyTransfer', $curlParams, $mensagem, $validacao);

			return redirect('transferencias')->with('warning', 'Erro ao realizar a operação. Por favor tente mais tarde.');
		}

		$transactionNumber = (isset($transfere['DocumentNumber'])) ? $transfere['DocumentNumber'] : '0';
		$sucesso = ($transfere["Success"] == "false" || $transfere["Success"] == "False") ? 0 : 1;
		$entryid = (isset($transfere["EntryId"])) ? $transfere["EntryId"] : '';
		$validacao = (isset($transfere['Validation'])) ? serialize($transfere['Validation']) : '';
		$mensagem = (isset($transfere['Message'])) ? $transfere['Message'] : '';
		$url = (isset($transfere['Url'])) ? $transfere['Url'] : '';
		$alreadyexists = ($transfere['AlreadyExists'] == "false" || $transfere['AlreadyExists'] == "False") ? 0 : 1;
		$limiteespecial = (isset($request->limiteespecial) && $request->limiteespecial == 1) ? '1' : '0';
		$valorlimiteespecial = 0;

		$checkSaved = checkDocNumberTransfer($transactionNumber);
		if (!$checkSaved) {
			$paymentDate = date2us($request->datatransf);

			$isScheduled = ($paymentDate == now()->format('Y-m-d')) ? 0 : 1;

			saveTransferLog(
				$client_taxnumber,
				somenteNumeros($request->documento),
				$request->nome,
				somenteNumeros($request->banco),
				somenteNumeros($request->agencia),
				somenteNumeros($request->conta),
				somenteNumeros($request->digito),
				somenteNumeros($request->tipoconta),
				$valor,
				$cli_ratevalue,
				$newIdentifier,
				$sucesso,
				$transactionNumber,
				$url,
				$alreadyexists,
				$saldoCompensar,
				$saldoCompensarValor,
				$curlParams,
				date2us($request->datatransf),
				$isScheduled,
			);
		}

		$saldoDisponivel = getAvailableLimitMultiCnpj($client_taxnumber);

		// if($logLimitUsed)
		//     logAvailableLimitUsed(Auth::user()->id, $client->id, $client_taxnumber, $saldoDisponivel, $saldoATransferir, 'TRANSFERENCIA', $transactionNumber, false, $internaltransferdocumentnumber, $cashbackvalue); // atualiza detalhamento

		// ATUALIZA GERENCIAMENTO DE TRANSFERENCIAS
		$transfermanagerid = (isset($request->transfermanagerid)) ? $request->transfermanagerid : 0;
		if ($transfermanagerid > 0) {
			$transferinfo = Transfermanager::where('id', '=', $transfermanagerid)->first();
			$transferinfo->status = 'Autorizado';
			$transferinfo->save();
		}

		if ($idAdiantamentoUtilizado) {
			$objAdiantamentoUtilizado = Usedavailablelimit::find($idAdiantamentoUtilizado);
			$objAdiantamentoUtilizado->documentnumber = $transactionNumber;
			$objAdiantamentoUtilizado->save();
		}

		return view('clients.transferenciabancariaok', compact('transfere'));
	}

	public function consultatransferenciabancaria(Request $request, $filtrostatus = null)
	{

		$datade = date('Y-m-d', strtotime('-7 days'));
		$dataate = date('Y-m-d');
		//$filtrostatus = 'Incluido';
		if (isset($request)) {
			if (isset($request->dataate)) {
				$dataate = date2us($request->dataate);
			}
			if (isset($request->datade) && (date2us($request->datade) <= $dataate)) {
				$datade = date2us($request->datade);
			}
			if (isset($request->filtrostatus)) {
				$filtrostatus = $request->filtrostatus;
			}
			//dd($request->datade, $datade, $request->dataate, $dataate);
		}

		$first = Transfermanager::select(
			'id',
			'url',
			'client_id',
			'user_id',
			'actiondate',
			'status',
			'documento',
			'banco',
			'agencia',
			'conta',
			'digito',
			'identificador',
			'valor',
			'documentnumber',
			'created_at',
		)
			->where('taxnumber', '=', Session::get('cnpj'))
			->whereDate('actiondate', '>=', $datade)
			->whereDate('actiondate', '<=', $dataate);
		if (isset($filtrostatus)) {
			$first->where('status', '=', $filtrostatus);
		}

		$q_tlogs = Transferlog::select(
			'id',
			'url',
			'client_id',
			'user_id',
			'actiondate',
			'status',
			'documento',
			'banco',
			'agencia',
			'conta',
			'digito',
			'identificador',
			'valor',
			'documentnumber',
			'created_at',
		)->where('user_id', '=', Auth::user()->id)
			->where('taxnumber', '=', Session::get('cnpj'))
			->whereDate('actiondate', '>=', $datade)
			->whereDate('actiondate', '<=', $dataate);
		//dd(isset($filtrostatus), $filtrostatus);
		if (isset($filtrostatus)) {
			$q_tlogs->where('status', '=', $filtrostatus);
		}
		$q_tlogs->union($first);

		$tlogs = $q_tlogs->orderBy('created_at', 'desc')->get();

		$datafiltro = array('datade' => date2br($datade), 'dataate' => date2br($dataate), 'filtrostatus' => $filtrostatus);

		return view('clients.logtransferencias', compact('tlogs', 'datafiltro'));

	}

	/* 20220408
																				      public function consultatransferenciabancaria_HELDER(Request $request) {
																					  if(isset($request) && $request->datade != ''  && $request->dataate != '' && ( $request->datade <= $request->dataate ) ) {
																					      $datade  = date(date2us($request->datade), strtotime('-7 days'));
																					      $dataate = date(date2us($request->dataate));
																					  } else {
																					      $datade  = date('Y-m-d', strtotime('-7 days'));
																					      $dataate = date('Y-m-d');
																					  }

																					  $client_taxnumber = Session::get('cnpj');

																					  $q_tlogs = Transferlog::where('user_id','=',Auth::user()->id)
																							      ->whereDate('actiondate','>=',$datade)
																							      ->whereDate('actiondate','<=',$dataate);

																					  //$q_tlogs = Transfermanager::where('taxnumber')

																					  if(isset($request) && $request->filtrostatus != '')
																					      $q_tlogs->where('status','=',$request->filtrostatus);


																					  $tlogs = $q_tlogs->orderBy('created_at','desc')->get();

																					  $filtrostatus  = (isset($request) && $request->filtrostatus != '')?$request->filtrostatus:'';

																					  $datafiltro = array('datade'=>date2br($datade),'dataate'=>date2br($dataate), 'filtrostatus'=>$filtrostatus);

																					  return view('clients.logtransferencias',compact('tlogs','datafiltro'));

																				      }
																				  */
	/* ********************** /TRANSFERENCIA BANCÁRIA ********************** */



	/* ################################### PAGAMENTOS ################################### */
	/*    public function pagamentoboleto(Request $request) {
																					  if ($request->isMethod('post')) {
																					      $curlParams = "{ \"Method\": \"GetInfosByBarcode\",
																							  \"PartnerId\": ".env('FITBANK_PARTNERID').",
																							  \"BusinessUnitId\": ".env('FITBANK_BUSINESSUNITID').",
																							  \"Barcode\": \"".trim(somenteNumeros($request->barcode))."\"}";

																					      $consulta = sendCurl($curlParams);

																					      if($consulta["Success"] == "false") {
																						  return redirect('/pagamentos/boletos')->with('warning', 'Erro ao consultar boleto. '.$consulta['Message']);
																					      }

																					      $boleto = json_decode($consulta['Infos']);

																					      return view('clients.pagamentoboleto',compact('boleto'));
																					  }
																					  return view('clients.pagamentoboletocodbarras');
																				      }
																				  */
	/*public function pagamentoboletopagar(Request $request) {
																				      $client = getClientById( Auth::user()->client_id );


																				      // CONSULTA O SALDO DO CLIENTE
																				      $saldoCliente = saldoCliente($client->taxnumber);
																				      $saldoDisponivel = $client->availablelimit;
																				      // valor a ser transferido
																				      $valor = covert2Float($request->valor);
																				      // verifica se o disponível é >= valor a ser transferido
																				      if($valor > $saldoCliente) {
																					  // se não for, verifica se permite usar saldo a compensar
																					  if(isset($request->limiteespecial) && $request->limiteespecial == 1) {
																					      // verifica se o cliente possui saldo a compensar suficiente
																					      if( ($saldoCliente + $saldoDisponivel) < $valor ) {
																						  return redirect('/pagamentos/boletos')->with('warning', 'Não há saldo suficiente para realizar a operação.');
																					      } else {
																						  // realiza a transferencia do valor adicional para o cliente
																						  $saldoATransferir = ($valor - $saldoCliente);

																						  $transfereSaldoDisponivel = transfereSaldoDisponivelCliente($client->taxnumber, $saldoATransferir, 'Transferencia de saldo a compensar');

																						  if(!$transfereSaldoDisponivel) return redirect('/pagamentos/boletos')->with('warning', 'Ocorreu um erro ao realizar a operação. Por favor tente mais tarde.');
																					      }
																					  } else {
																					      return redirect('/pagamentos/boletos')->with('warning', 'Não há saldo suficiente para realizar a operação.');
																					  }
																				      }

																				      $spsInfo = getSPS();

																				      $vencimento = (!empty($request->duedate))?dataSlashReverse($request->duedate):date('Y/m/d');

																				      $curlParams = "{\"Method\": \"GenerateBoletoOut\",
																						  \"PartnerId\": ".env('FITBANK_PARTNERID').",
																						  \"BusinessUnitId\": ".env('FITBANK_BUSINESSUNITID').",
																						  \"Name\": \"".$client->name."\",
																						  \"TaxNumber\": ".$client->taxnumber.",
																						  \"Barcode\": \"".$request->barcode."\",
																						  \"PrincipalValue\": ".$request->principalvalue.",
																						  \"DiscountValue\": ".str_replace(',', '.', str_replace('.', '', $request->desconto)).",
																						  \"ExtraValue\": ".str_replace(',', '.', str_replace('.', '', $request->juros)).",
																						  \"PaymentDate\": \"".$request->datapagamento."\",
																						  \"DueDate\": \"".$vencimento."\",
																						  \"Identifier\": \"".$request->identificador."\",
																						  \"RateValueType\": 1,
																						  \"RateValue\": ".$spsInfo->ratevalue."
																					      }";

																				      $pagamento = sendCurl($curlParams);

																				      if($pagamento["Success"] == "false") {
																					  return redirect('/pagamentos/boletos')->with('warning', 'Erro ao realizar a operação. '.$pagamento['Message']);
																				      }

																				      $transactionNumber = (isset($pagamento['DocumentNumber']))?$pagamento['DocumentNumber']:'0';
																				      $sucesso = ($pagamento["Success"] == "false")?0:1;
																				      $entryid = (isset($pagamento["EntryId"]))?$pagamento["EntryId"]:'';
																				      $validacao = (isset($pagamento['Validation']))?serialize($pagamento['Validation']):'';
																				      $mensagem = (isset($pagamento['Message']))?$pagamento['Message']:'';
																				      $url = (isset($pagamento['Url']))?$pagamento['Url']:'';

																				      savePaymentLog($client->taxnumber ,str_replace('/', '-', $vencimento) ,str_replace('/', '-', $request->datapagamento) ,'GenerateBoletoOut' ,
																						      $request->identificador , $transactionNumber, $request->digitableline ,$curlParams ,$sucesso ,$mensagem ,$validacao ,
																						      $entryid ,$url ,$request->principalvalue ,str_replace(',', '.', str_replace('.', '', $request->desconto)) ,str_replace(',', '.', str_replace('.', '', $request->juros)));

																				      return view('clients.pagamentoboletook', compact('pagamento'))->with('success', 'Operação realizada com sucesso.');
																				  }

																				  public function logpagamentos(Request $request) {

																				      if(isset($request) && $request->datade != ''  && $request->dataate != '' && ( $request->datade <= $request->dataate ) ) {
																					  $datade  = date(date2us($request->datade), strtotime('-7 days'));
																					  $dataate = date(date2us($request->dataate));
																				      } else {
																					  $datade  = date('Y-m-d', strtotime('-7 days'));
																					  $dataate = date('Y-m-d');
																				      }

																				      $tlogs = Paymentlog::where('method','=','GenerateBoletoOut')
																							  ->where('user_id','=',Auth::user()->id)
																							  ->whereDate('actiondate','>=',$datade)
																							  ->whereDate('actiondate','<=',$dataate)
																							  ->where('url','<>','')
																							  ->orderBy('created_at','desc')
																							  ->get();

																				      $datafiltro = array('datade'=>dataSlashReverse($datade),'dataate'=>dataSlashReverse($dataate));

																				      return view('clients.logpagamentos',compact('tlogs','datafiltro'));
																				  }
																				  */
	/* ################################### PAGAMENTOS ################################### */
	public function criarBoletoAvulso()
	{
		$client = Client::where('id', '=', Auth::user()->client_id)->first();
		return view('clients.boleto_avulso', compact('client'));
	}

	public function criarBoletoContato(Request $request)
	{
		$client = Client::where('id', '=', Auth::user()->client_id)->first();
		$contact = Clientcontact::where('id', '=', $request->idcontato)->first();
		return view('clients.boleto_contato', compact('client', 'contact'));
	}



	public function criarBoleto(Request $request)
	{

		// ********** MktPlaceId master ou identifier do cliente??????????
		$client = getClientById(Auth::user()->client_id);

		$spsInfo = getSPS();
		$datavencimento = dataSlashReverse($request->DueDate);
		$datamulta = ($request->FineDate != '') ? dataSlashReverse($request->FineDate) : '';
		$datadesconto = ($request->DiscountDate != '') ? dataSlashReverse($request->DiscountDate) : '';

		$newIdentifier = concatIdentifier($request->Identifier);

		$cli_ratevalue = consultaTaxaCliente($client->id, 'boletoout', covert2Float($request->TotalValue));

		$curlParams = "{
                    \"Method\": \"GenerateBoleto\",
                    \"MktPlaceId\": " . $spsInfo->mktplaceid . ",
                    \"PartnerId\": " . $spsInfo->partnerid . ",
                    \"BusinessUnitId\": " . $spsInfo->businessunitid . ",
                    \"GroupTemplate\": 2,
                    \"CustomerName\": \"" . $request->CustomerName . "\",
                    \"CustomerTaxNumber\": \"" . somenteNumeros($request->CustomerTaxNumber) . "\",
                    \"CustomerMail\": \"" . $request->CustomerMail . "\",
                    \"CustomerPhone\": \"" . somenteNumeros($request->CustomerPhone) . "\",
                    \"RateValue\": \"" . $cli_ratevalue . "\",
                    \"RateSent\": \"1\",
                    \"SupplierTaxNumber\": \"" . somenteNumeros($request->SupplierTaxNumber) . "\",
                    \"SupplierFullName\": \"" . $request->SupplierFullName . "\",
                    \"SupplierTradingName\": \"" . $request->SupplierTradingName . "\",
                    \"SupplierLegalName\": \"" . $request->SupplierLegalName . "\",
                    \"SupplierMail\": \"" . $request->SupplierMail . "\",
                    \"SupplierPhone\":\"" . somenteNumeros($request->SupplierPhone) . "\",
                    \"AddressLine1\": \"" . $request->AddressLine1 . "\",
                    \"AddressLine2\": \"" . $request->AddressLine2 . "\",
                    \"Neighborhood\": \"" . $request->Neighborhood . "\",
                    \"City\": \"" . $request->City . "\",
                    \"State\": \"" . $request->State . "\",
                    \"ZipCode\": \"" . somenteNumeros($request->ZipCode) . "\",
                    \"MailToSend\": \"" . $request->MailToSend . "\",
                    \"PhoneToSend\": \"" . somenteNumeros($request->PhoneToSend) . "\",
                    \"ExternalNumber\": \"" . $request->ExternalNumber . "\",
                    \"Identifier\": \"" . $newIdentifier . "\",
                    \"Comments\": \"" . $request->Comments . "\",
                    \"DueDate\": \"" . $datavencimento . "\",
                    \"TotalValue\": " . covert2Float($request->TotalValue) . ",
                    \"FineDate\": \"" . $datamulta . "\",
                    \"FinePercent\": \"" . covert2Float($request->FinePercent) . "\",
                    \"FineValue\": \"" . covert2Float($request->FineValue) . "\",
                    \"InterestPercent\": \"" . covert2Float($request->InterestPercent) . "\",
                    \"InterestValue\": \"" . covert2Float($request->InterestValue) . "\",
                    \"DiscountDate\": \"" . $datadesconto . "\",
                    \"DiscountValue\": \"" . covert2Float($request->DiscountValue) . "\",
                    \"RebateValue\": \"" . covert2Float($request->RebateValue) . "\",
                    \"Products\":[{
                        \"SellerPersonType\": \"" . $request->SellerPersonType . "\",
                        \"SellerName\": \"" . $request->SellerName . "\",
                        \"SellerTaxNumber\": \"" . somenteNumeros($request->SellerTaxNumber) . "\",
                        \"ReceiverPersonType\": \"" . $request->ReceiverPersonType . "\",
                        \"ReceiverName\": \"" . $request->ReceiverName . "\",
                        \"ReceiverTaxNumber\": \"" . somenteNumeros($request->ReceiverTaxNumber) . "\",
                        \"Reference\": \"" . $request->Reference . "\",
                        \"ProductCode\": \"" . $request->ProductCode . "\",
                        \"ProductName\": \"" . $request->ProductName . "\",
                        \"ProductQty\": \"" . $request->ProductQty . "\",
                        \"ProductValue\": \"" . covert2Float($request->ProductValue) . "\",
                        \"ProductRateValueToTransfer\": \"" . covert2Float($request->ProductRateValueToTransfer) . "\"
                    }]
                }";

		$boleto = sendCurl($curlParams);

		if ($boleto["Success"] == "false") {
			$validacao = (isset($boleto['Validation'])) ? serialize($boleto['Validation']) : '';


			$logerror = saveBilleterrorLog(
				somenteNumeros($request->SupplierTaxNumber),
				$request->CustomerName,
				somenteNumeros($request->CustomerTaxNumber),
				covert2Float($request->TotalValue),
				$boleto['Message'],
				$curlParams,
				$validacao
			);

			return redirect('/clientes/boletos/avulso')->with('warning', 'Erro ao realizar a operação. ' . $boleto['Message']);
		}

		$clientcontact_id = 0;
		$transactionNumber = (isset($boleto['DocumentNumber'])) ? $boleto['DocumentNumber'] : '0';
		$sucesso = ($boleto["Success"] == "false") ? 0 : 1;
		$entryid = (isset($boleto["EntryId"])) ? $boleto["EntryId"] : '';
		$validacao = (isset($boleto['Validation'])) ? serialize($boleto['Validation']) : '';
		$mensagem = (isset($boleto['Message'])) ? $boleto['Message'] : '';
		$url = (isset($boleto['Url'])) ? $boleto['Url'] : '';
		$alreadyexists = (isset($transfere['AlreadyExists'])) ? 1 : 0;
		$boletoinfo = (isset($boleto['Boleto'])) ? serialize($boleto['Boleto']) : '';


		$salvaboleto = saveBillet(
			$clientcontact_id,
			somenteNumeros($request->SupplierTaxNumber),
			$request->CustomerName,
			somenteNumeros($request->CustomerTaxNumber),
			$request->CustomerMail,
			somenteNumeros($request->CustomerPhone),
			$request->AddressLine1,
			$request->AddressLine2,
			$request->Neighborhood,
			$request->City,
			$request->State,
			somenteNumeros($request->ZipCode),
			$request->MailToSend,
			somenteNumeros($request->PhoneToSend),
			$request->ExternalNumber,
			$newIdentifier,
			$request->Comments,
			$request->DueDate,
			covert2Float($request->TotalValue),
			$request->FineDate,
			covert2Float($request->FinePercent),
			covert2Float($request->FineValue),
			covert2Float($request->InterestPercent),
			covert2Float($request->InterestValue),
			$request->DiscountDate,
			covert2Float($request->DiscountValue),
			covert2Float($request->RebateValue),
			$transactionNumber,
			$entryid,
			$url,
			$alreadyexists,
			$curlParams,
			$boletoinfo,
			$cli_ratevalue
		);


		return redirect('/clientes/boletos/consultar')->with('status', 'Boleto criado.');
	}

	public function consultarBoleto(Request $request)
	{

		if (isset($request) && $request->datade != '' && $request->dataate != '' && ($request->datade <= $request->dataate)) {
			$datade = date2us($request->datade, strtotime('-15 days'));
			$dataate = date2us($request->dataate);
		} else {
			$datade = date('Y-m-d', strtotime('-15 days'));
			$dataate = date('Y-m-d');
		}

		$q_billets = Billet::where('client_id', '=', Auth::user()->client_id)
			->where('alreadyexists', '=', 0)
			->whereDate('emission', '>', $datade)
			->whereDate('emission', '<=', $dataate);

		if (isset($request) && $request->filtrostatus != '')
			$q_billets->where('status', '=', $request->filtrostatus);


		$billets = $q_billets->orderBy('created_at', 'desc')->get();

		$filtrostatus = (isset($request) && $request->filtrostatus != '') ? $request->filtrostatus : '';

		$datafiltro = array('datade' => dataUs2Br($datade), 'dataate' => dataUs2Br($dataate), 'filtrostatus' => $filtrostatus);

		return view('clients.boleto_consultar', compact('billets', 'datafiltro'));
	}

	/* ************************ CONTATOS *****************************/
	public function contatosCliente(Request $request)
	{
		if ($request->isMethod('post') && $request->nomecontato != '') {
			$contacts = Clientcontact::where('client_id', '=', Auth::user()->client_id)
				->where('customername', 'like', '%' . $request->nomecontato . '%')
				->orderBy('customername', 'ASC')
				->get();
			$filtro = array('nomecontato' => $request->nomecontato);
		} else {
			$contacts = Clientcontact::where('client_id', '=', Auth::user()->client_id)->orderBy('customername', 'ASC')->get();
			$filtro = array('nomecontato' => '');
		}

		return view('clients.contatos', compact('contacts', 'filtro'));
	}

	public function contatosClienteCriar(Request $request)
	{
		if ($request->isMethod('post')) {
			$contact = new Clientcontact();

			$contact->client_id = Auth::user()->client_id;
			$contact->customername = $request->customername;
			$contact->customertaxnumber = somenteNumeros($request->customertaxnumber);
			$contact->customermail = $request->customermail;
			$contact->customerphone = somenteNumeros($request->customerphone);
			$contact->addressline1 = $request->addressline1;
			$contact->addressline2 = $request->addressline2;
			$contact->neighborhood = $request->neighborhood;
			$contact->city = $request->city;
			$contact->state = $request->state;
			$contact->zipcode = $request->zipcode;
			$contact->bankcode = $request->bankcode;
			$contact->bank = $request->bank;
			$contact->bankbranch = $request->bankbranch;
			$contact->bankaccount = $request->bankaccount;
			$contact->bankaccountdigito = $request->bankaccountdigito;

			$contact->save();
			return redirect('/contatos/cliente')->with('status', 'Contato salvo!');
		}

		return view('clients.contatos_criar');
	}

	public function atualizasaldodisponivel(Request $request)
	{
		$client = Client::select('id', 'availablelimit')->where('taxnumber', '=', $request->taxnumber)->first();

		if (!$client) {
			$multi = Multisafe::where('taxnumber', '=', $request->taxnumber)->first();
			$multi->availablelimit = covert2Float($request->valor);
			$multi->update();
		} else {
			$client->availablelimit = covert2Float($request->valor);
			$client->update();
		}

		if (!$client && !$multi)
			return redirect('clientes/' . $request->client_id . '/editar')->with('warning', 'Algo aconteceu. Por favor tente mais tarde.');

		return redirect('clientes/' . $request->client_id . '/editar')->with('success', 'Atualizado com sucesso.');
	}

	/**
	 * Atualiza os dados do pagamento da mensalidade do cliente.
	 * @param Request $request Dados da view
	 * @return \Illuminate\Routing\Redirector|\Illuminate\Http\RedirectResponse
	 */
	public function atualizarDadosMensalidade(Request $request)
	{
		$clienteObj = buscarClienteOuMultisafePeloTaxnumber($request->taxnumber);

		if ($request->has('valor'))
			$clienteObj->monthlyfee = covert2Float($request->valor);
		if ($request->has('monthlyfeedueday'))
			$clienteObj->monthlyfeedueday = $request->monthlyfeedueday;
		$clienteObj->update();

		if (!$clienteObj)
			return redirect('clientes/' . $request->client_id . '/editar')->with('warning', 'Algo aconteceu. Por favor tente mais tarde.');

		return redirect('clientes/' . $request->client_id . '/editar')->with('success', 'Atualizado com sucesso.');
	}

	public function informacoes(Request $request, $id)
	{
		$client = Client::where('id', '=', Auth::user()->client_id)->first();

		return view('clients.informacoes', compact('client'));
	}

	public function lancamentosFuturos($cliente)
	{
		$hoje = date('Y-m-d');

		$darf = Darfpayment::select('id', 'documentnumber', 'totalvalue as value', 'paymentdate', DB::raw('"Darf" as type'))
			->where('client_id', '=', $cliente)
			->where('status', '=', '0')
			->whereDate('paymentdate', '>', $hoje)
			->get();

		$darj = Darjpayment::select('id', 'documentnumber', 'totalvalue as value', 'paymentdate', DB::raw('"Darj" as type'))
			->where('client_id', '=', $cliente)
			->where('status', '=', '0')
			->whereDate('paymentdate', '>', $hoje)
			->get();

		$fgts = Fgtspayment::select('id', 'documentnumber', 'principalvalue as value', 'paymentdate', DB::raw('"Fgts" as type'))
			->where('client_id', '=', $cliente)
			->where('status', '=', '0')
			->whereDate('paymentdate', '>', $hoje)
			->get();

		$gps = Gpspayment::select('id', 'documentnumber', 'principalvalue as value', 'paymentdate', DB::raw('"Gps" as type'))
			->where('client_id', '=', $cliente)
			->where('status', '=', '0')
			->whereDate('paymentdate', '>', $hoje)
			->get();

		$gare = Garepayment::select('id', 'documentnumber', 'totalvalue as value', 'paymentdate', DB::raw('"Gare" as type'))
			->where('client_id', '=', $cliente)
			->where('status', '=', '0')
			->whereDate('paymentdate', '>', $hoje)
			->get();

		$barcodepayment = Barcodepayment::select('id', 'documentnumber', 'totalvalue as value', 'paymentdate', DB::raw('"Pagamento" as type'))
			->where('client_id', '=', $cliente)
			->where('status', '=', '0')
			->whereDate('paymentdate', '>', $hoje)
			->get();

		$transfer = Transferlog::select('id', 'documentnumber', 'valor as value', 'paymentdate', DB::raw('"Transferência" as type'))
			->where('client_id', '=', $cliente)
			->where('status', '=', '0')
			->whereDate('paymentdate', '>', $hoje)
			->get();


		$availablelimit = Usedavailablelimit::select('id', 'documentnumber', 'interestvalue as value', DB::raw('"Adiant. saldo" as type'), DB::raw('"Adiant. saldo" as type'))
			->where('client_id', '=', $cliente)
			->where('paid', '=', '0')
			->get();




		$futuros = $darf->concat($darj)->concat($fgts)->concat($gps)->concat($gare)->concat($barcodepayment)->concat($transfer)->concat($availablelimit);

		return $futuros->sortBy('paymentdate');
	}

	/* operações pelo cliente */
	public function operacoesPeloCliente(Request $request, $id)
	{
		$client = Client::where('id', '=', $id)->first();

		$spsInfo = getSPS();

		$curlParams = "{ \"Method\": \"GetBanks\",
                    \"PartnerId\": " . $spsInfo->partnerid . ",
                    \"BusinessUnitId\": " . $spsInfo->businessunitid . "
                }";

		$buscabancos = sendCurl($curlParams);

		$bancos = ($buscabancos["Success"] == "false") ? false : $buscabancos["Banks"];

		$datade = date('Y/m/d', strtotime('-7 days'));
		$dataate = date('Y/m/d');
		$curlParams = "{ \"Method\": \"GetAccountEntry\",
                            \"PartnerId\": " . $spsInfo->partnerid . ",
                            \"BusinessUnitId\": " . $spsInfo->businessunitid . ",
                            \"TaxNumber\": \"" . $client->taxnumber . "\",
                            \"StartDate\": \"" . $datade . "\",
                            \"EndDate\": \"" . $dataate . "\"
                        }";

		$extrato = sendCurl($curlParams);
		$msg = false;
		if ($extrato["Success"] == "false") {
			$msg = 'Ocorreu um problema ao solicitar o saldo SPS.';
			$extrato['Balance'] = 0;
		}
		/* /consultar saldo atual do cliente */

		return view('clients.operacoes', compact('bancos', 'extrato', 'client'))->with('status', $msg);
	}

	/* realizar operações pelo cliente */
	public function operacoesPeloClienteRealizar(Request $request, $id)
	{
		$client = getClientById($id);

		$spsInfo = getSPS();


		if (!operationTimeLimit('transferencia') && isToday($request->datatransf)) {
			return redirect('transferencias')->with('warning', 'Transferências após o horário limite devem ser agendadas para o próximo dia útil.');
		}

		if (isWeekend($request->datatransf)) {
			return redirect('transferencias')->with('warning', 'Transferências nos finais de semana devem ser agendadas para o próximo dia útil.');
		}


		// CONSULTA O SALDO DO CLIENTE
		$saldoCliente = saldoCliente($client->taxnumber);
		$saldoDisponivel = $client->availablelimit;
		// valor a ser transferido
		$valor = covert2Float($request->valor);
		$logLimitUsed = false;

		$saldoCompensar = 0;
		$saldoCompensarValor = 0;
		$internaltransferdocumentnumber = 0;
		$cashbackvalue = 0;

		$cli_ratevalue = consultaTaxaCliente($client->id, 'moneytransfer', covert2Float($valor));
		// ratevaluetype:  0 - Enviado (Taxa enviada na requisição); 1 - Padrão (Taxa configurada na Unidade de negócio);2 - Nenhum (Sem Taxa)

		// verifica se o disponível é >= valor a ser transferido
		if ($valor > $saldoCliente) {
			// se não for, verifica se permite usar saldo a compensar
			if (isset($request->limiteespecial) && $request->limiteespecial == 1) {

				// saldo a transferir para completar o pagamento
				$saldoATransferir = ($valor - $saldoCliente);

				// ao saldo a transferir deve ser adicionada a taxa
				$cashbackvalue = consultaTaxaCliente($client->id, 'internaltransfer', $saldoATransferir);

				// verifica se o cliente possui saldo a compensar suficiente
				if (($saldoCliente + $saldoDisponivel) < ($valor + $cashbackvalue + $cli_ratevalue)) {
					return redirect('transferencias')->with('warning', 'Não há saldo suficiente para realizar a operação.');
				} else {
					// realiza a transferencia do valor adicional para o cliente
					$transfereSaldoDisponivel = transfereSaldoDisponivelCliente($client->taxnumber, $saldoATransferir, 'Transferencia de saldo a compensar', $cashbackvalue);

					if (!$transfereSaldoDisponivel)
						return redirect('transferencias')->with('warning', 'Ocorreu um erro ao realizar a operação. Por favor tente mais tarde.');

					$logLimitUsed = true;
					$saldoCompensar = 1;
					$saldoCompensarValor = $saldoATransferir;
					$internaltransferdocumentnumber = $transfereSaldoDisponivel;
				}
			} else {
				return redirect('transferencias')->with('warning', 'Não há saldo suficiente para realizar a operação.');
			}
		}
		// /CONSULTA O SALDO DO CLIENTE

		$newIdentifier = concatIdentifier($request->identificador);

		$curlParams = "{ \"Method\": \"MoneyTransfer\",
                    \"PartnerId\": " . $spsInfo->partnerid . ",
                    \"BusinessUnitId\": " . $spsInfo->businessunitid . ",
                    \"FromTaxNumber\": \"" . $client->taxnumber . "\",
                    \"ToTaxNumber\": \"" . somenteNumeros($request->documento) . "\",
                    \"ToName\": \"" . $request->nome . "\",
                    \"Bank\": \"" . somenteNumeros($request->banco) . "\",
                    \"BankBranch\": \"" . somenteNumeros($request->agencia) . "\",
                    \"BankAccount\": \"" . somenteNumeros($request->conta) . "\",
                    \"BankAccountDigit\": \"" . somenteNumeros($request->digito) . "\",
                    \"AccountType\": \"" . somenteNumeros($request->tipoconta) . "\",
                    \"Value\": " . $valor . ",
                    \"RateValue\": \"" . $cli_ratevalue . "\",
                    \"Identifier\": \"" . $newIdentifier . "\",
                    \"RateValueType\": \"0\",
                    \"PaymentDate\": \"" . dataSlashReverse($request->datatransf) . "\",
                    \"Tags\": []
                }";

		customLog('realizatransferenciabancaria: ' . $curlParams);

		$transfere = sendCurl($curlParams);

		if ($transfere["Success"] == "false") {
			$validacao = (isset($transfere['Validation'])) ? serialize($transfere['Validation']) : '';
			$mensagem = (isset($transfere['Message'])) ? $transfere['Message'] : '';

			saveTransferErrorLog('MoneyTransfer', $curlParams, $mensagem, $validacao);

			return redirect('/clientes/' . $id . '/operacoes')->with('warning', 'Erro ao realizar a operação. Por favor tente mais tarde.');
		}

		$transactionNumber = (isset($transfere['DocumentNumber'])) ? $transfere['DocumentNumber'] : '0';
		$sucesso = ($transfere["Success"] == "false") ? 0 : 1;
		$entryid = (isset($transfere["EntryId"])) ? $transfere["EntryId"] : '';
		$validacao = (isset($transfere['Validation'])) ? serialize($transfere['Validation']) : '';
		$mensagem = (isset($transfere['Message'])) ? $transfere['Message'] : '';
		$url = (isset($transfere['Url'])) ? $transfere['Url'] : '';
		$alreadyexists = (isset($transfere['AlreadyExists'])) ? 1 : 0;

		$limiteespecial = (isset($request->limiteespecial) && $request->limiteespecial == 1) ? '1' : '0';
		$valorlimiteespecial = 0;

		$checkSaved = checkDocNumberTransfer($transactionNumber);
		if (!$checkSaved)
			saveTransferLog(
				$client->taxnumber,
				somenteNumeros($request->documento),
				$request->nome,
				somenteNumeros($request->banco),
				somenteNumeros($request->agencia),
				somenteNumeros($request->conta),
				somenteNumeros($request->digito),
				somenteNumeros($request->tipoconta),
				$valor,
				$cli_ratevalue,
				$newIdentifier,
				$sucesso,
				$transactionNumber,
				$url,
				$alreadyexists,
				$saldoCompensar,
				$saldoCompensarValor,
				$curlParams,
				date2us($request->datatransf)
			);

		if ($logLimitUsed)
			logAvailableLimitUsed(Auth::user()->id, $client->id, $client->taxnumber, $client->availablelimit, $saldoATransferir, 'TRANSFERENCIA', $transactionNumber, false, $internaltransferdocumentnumber, $cashbackvalue); // atualiza detalhamento

		return redirect('/clientes/' . $id . '/operacoes')->with('success', 'Operação realizada com sucesso.');
	}

	/* ver o extrato do cliente */
	public function operacoesPeloClienteExtrato(Request $request, $id)
	{
		$client = getClientById($id);

		$spsInfo = getSPS();

		if (isset($request) && $request->datade != '' && $request->dataate != '' && ($request->datade <= $request->dataate)) {
			$datade = date(dataSlashReverse($request->datade), strtotime('-15 days'));
			$dataate = date(dataSlashReverse($request->dataate));
		} else {
			$datade = date('Y/m/d', strtotime('-15 days'));
			$dataate = date('Y/m/d');
		}

		$curlParams = "{ \"Method\": \"GetAccountEntry\",
                    \"PartnerId\": " . $spsInfo->partnerid . ",
                    \"BusinessUnitId\": " . $spsInfo->businessunitid . ",
                    \"TaxNumber\": \"" . $client->taxnumber . "\",
                    \"StartDate\": \"" . $datade . "\",
                    \"EndDate\": \"" . $dataate . "\"
                }";

		$extrato = sendCurl($curlParams);

		if ($extrato["Success"] == "false")
			return redirect('/home')->with('warning', 'Erro: ocorreu um problema ao processar a requisição.');

		$datafiltro = array('datade' => dataSlashReverse($datade), 'dataate' => dataSlashReverse($dataate));

		$lancamentosfuturos = $this->lancamentosFuturos($client->id);

		// $saldocompensar = getSmartsafeBalance($client->id);
		$saldocompensar = $client->availablelimit;

		return view('clients.extrato_cliente', compact('extrato', 'datafiltro', 'client', 'lancamentosfuturos', 'saldocompensar'));
	}

	public function changecnpj(Request $request)
	{
		if (Auth::user()->role_id > 1) {
			$client = getClientById(Auth::user()->client_id);
			Session::put('cnpj', $request->alteraconta);
			Session::put('cnpjformatado', formataCNPJ($request->alteraconta));
		}
		return redirect($request->currenturi)->with('success', 'Conta ativa alterada para: ' . formataCNPJ($request->alteraconta));
	}

	public function crianovaconta(Request $request)
	{
		$checkExist = Multisafe::where('taxnumber', '=', somenteNumeros($request->taxnumber))->first();

		if ($checkExist)
			return redirect('/clientes/' . $request->client_id . '/editar')->with('warning', 'Já existe uma conta com esse CNPJ ' . formataCNPJ(somenteNumeros($request->taxnumber)));

		$originalNickname = $request->nickname;
		$normalized = Normalizer::normalize($originalNickname, Normalizer::NFD);
		$nickname = preg_replace('/[[`~´^]/', '', preg_replace('/[\x{0300}-\x{036F}]/u', '', $normalized));

		$novaconta = new Multisafe();

		$novaconta->client_id = $request->client_id;
		$novaconta->maintaxnumber = somenteNumeros($request->maintaxnumber);
		$novaconta->taxnumber = somenteNumeros($request->taxnumber);
		$novaconta->active = 1;
		$novaconta->businessunitid = 0;
		$novaconta->mail = $request->mail;
		$novaconta->phone = $request->phone;
		$novaconta->nickname = $nickname;
		$novaconta->availablelimit = 0;

		$novaconta->save();

		return redirect('/clientes/' . $request->client_id . '/editar')
			->with('success', 'Conta criada com sucesso: ' . formataCNPJ(somenteNumeros($request->taxnumber)) . ' - ' . $novaconta->nickname);
	}

	public function multicontaMudastatus(Request $request, $id)
	{
		$multi = Multisafe::where('id', '=', $id)->first();

		try {

			if ($multi->d0automatic == "1") {
				$this->desativarD0Automatico($request["client_id"], $id);
			}

		} catch (\Exception $e) {
			// TODO
			// Checar com Robson caso não consiga desativar as operações
		}

		try {

			if ($multi->d1automatic == "1") {
				$this->desativarD1Automatico($request["client_id"], $id);
			}

		} catch (\Exception $e) {
			// TODO
			// Checar com Robson caso não consiga desativar as operações
		}

		$multi->active = ($multi->active == 1) ? 0 : 1;
		$multi->save();

		$message = ($multi->active == 1) ? 'ativada' : 'desativada';
		return redirect('/clientes/' . $request->client_id . '/editar')->with('success', 'Conta ' . $message . ' com sucesso: ' . formataCNPJ(somenteNumeros($request->taxnumber)));
	}

	public function atualizarStatusAgendamento($transferLog)
	{

		$spsInfo = getSPS();

		$curlParams = "{ \"Method\": \"GetMoneyTransferOutById\",
			\"PartnerId\": " . $spsInfo->partnerid . ",
			\"BusinessUnitId\": " . $spsInfo->businessunitid . ",
			\"DocumentNumber\": \"" . $transferLog->documentnumber . "\"
		}";

		$consulta = sendCurl($curlParams);

		if ($consulta["Success"] == "false") {
			return 'ERROR';
		}

		$transferencia = json_decode($consulta['Transferencia']);

		if ($transferLog->scheduled_status == $transferencia->Status) {
			return 'NOT_UPDATED';
		}

		$transferLog->scheduled_status = $transferencia->Status;
		$transferLog->save();

		return 'UPDATED';
	}

	public function agendamentos(Request $request)
	{
		// 1. VERIFICA SE O USER POSSUI AGENDAMENTOS
		// SE NAO POSSUIR, RETORNA DE ONDE VEIO

		$transfers = Transferlog::where('is_scheduled', '=', 1)
			->where('scheduled_status', '=', 0)
			->where('taxnumber', '=', Session::get('cnpj'))
			->get();

		if ($transfers->count() <= 0) {
			if($request->headers->get('referer') == route('transferencias.agendamento.consultar')){
				return redirect('/home')
				->with('success', 'Agendamento cancelado com sucesso. Você não possui transferências agendadas.');
			}
			return redirect('/home')
				->with('warning', 'Você não possui transferências agendadas');
		}

		// 2. ATUALIZAR O STATUS DE CADA AGENDAMENTO NÃO PROCESSADO
		$errorOnUpdate = false;
		foreach ($transfers as $k => $transfer) {
			$statusUpdated = $this->atualizarStatusAgendamento($transfer);

			if ($statusUpdated === 'UPDATED') {
				unset($transfers[$k]);
			}

			if ($statusUpdated === 'ERROR') {
				$errorOnUpdate = true;
				$transfer = $transfer->toArray();
				$transfer['errorOnUpdate'] = true;
				$transfers[$k] = $transfer;
			}
		}

		// 3. CARREGA A VIEW
		return view('clients.transferencia.agendamentos', [
			'transfers' => $transfers->toArray(),
			'errorOnUpdate' => $errorOnUpdate,
		]);
	}

	public function cancelarAgendamento(Request $request)
	{
		$spsInfo = getSPS();

		$curlParams = "{ \"Method\": \"CancelMoneyTransfer\",
                    \"PartnerId\": " . $spsInfo->partnerid . ",
                    \"BusinessUnitId\": " . $spsInfo->businessunitid . ",
                    \"DocumentNumber\": \"" . $request->documentNumber . "\"
        }";

		$consulta = sendCurl($curlParams);

		if ($consulta["Success"] == "false") {
			return redirect(route('transferencias.agendamento.consultar'))
				->with('warning', 'Ocorreu um erro ao cancelar o agendamento');
		}

		$updateTransfer = Transferlog::where('is_scheduled', '=', 1)
			->where('scheduled_status', '=', 0)
			->where('taxnumber', '=', Session::get('cnpj'))
			->where('documentnumber', '=', $request->documentNumber)
			->update([
				'scheduled_status' => 5,
				'cancellation_date' => date('Y-m-d H:i:s')

			]);

		return redirect(route('transferencias.agendamento.consultar'))
			->with('success', 'Agendamento cancelado com sucesso');
	}
}
